<?php
/**
 * Mark Course
 *
 * @package     AutomatorWP\Integrations\Tutor_LMS\Actions\Mark_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Tutor_LMS_Mark_Course extends AutomatorWP_Integration_Action {

    public $integration = 'tutor';
    public $action = 'tutor_mark_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Mark course as completed for the user', 'automatorwp-pro' ),
            'select_option'     => __( 'Mark <strong>course as completed</strong> for the user', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. */
            'edit_label'        => sprintf( __( 'Mark %1$s as completed for the user', 'automatorwp-pro' ), '{post}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'Mark %1$s as completed for the user', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_default'    => __( 'course', 'automatorwp-pro' ),
                    'placeholder'       => __( 'Select a course', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type'         => apply_filters( 'tutor_course_post_type', 'courses' ),
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $course_id = absint( $action_options['post'] );

        // Bail if not course provided
        if( $course_id === 0 ) {
            return;
        }

        // Bail if user already completed this course
        if ( tutils()->is_completed_course( $course_id, $user_id ) ) {
            return;
        }

        $completion_process = tutils()->get_option( 'course_completion_process' );

        // If completion process is set to strict, then mark all lessons as completed too
        if ( $completion_process === 'strict') {

            $lesson_query = tutils()->get_lesson( $course_id, -1 );

            if ( $lesson_query->found_posts > 0 ) {
                foreach ( $lesson_query->posts as $lesson ) {
                    // Mark lesson as completed
                    tutils()->mark_lesson_complete( $lesson->ID, $user_id );
                }
            }

        }

        // Mark course as completed
        $this->complete_course_for_user( $user_id, $course_id );

    }

    public function complete_course_for_user( $user_id, $course_id ) {

        global $wpdb;

        // Trigger action for compatibility
        do_action( 'tutor_course_complete_before', $course_id );

        $date = date( "Y-m-d H:i:s", tutor_time() );

        // Generate an unique hash
        do {
            $hash = substr( md5( wp_generate_password( 32 ) . $date . $course_id . $user_id ), 0, 16 );
            $has_hash = (int) $wpdb->get_var( "SELECT COUNT(comment_ID) from {$wpdb->comments} WHERE comment_agent = 'TutorLMSPlugin' AND comment_type = 'course_completed' AND comment_content = '{$hash}' " );
        } while ( $has_hash > 0 );

        $data = array(
            'comment_post_ID'  => $course_id,
            'comment_author'   => $user_id,
            'comment_date'     => $date,
            'comment_date_gmt' => get_gmt_from_date( $date ),
            'comment_content'  => $hash,
            'comment_approved' => 'approved',
            'comment_agent'    => 'TutorLMSPlugin',
            'comment_type'     => 'course_completed',
            'user_id'          => $user_id,
        );

        $wpdb->insert( $wpdb->comments, $data );

        // Trigger action for compatibility
        do_action( 'tutor_course_complete_after', $course_id );

    }

}

new AutomatorWP_Tutor_LMS_Mark_Course();