<?php
/**
 * Tags
 *
 * @package     AutomatorWP\Integrations\SliceWP\Tags
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Commission tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_slicewp_commission_tags() {

    $commission_tags = array(
        'slicewp_commission_id' => array(
            'label'     => __( 'Commission ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '123',
        ),
        'slicewp_commission_amount' => array(
            'label'     => __( 'Commission amount', 'automatorwp-pro' ),
            'type'      => 'float',
            'preview'   => '123.45',
        ),
        'slicewp_commission_currency' => array(
            'label'     => __( 'Commission currency', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'USD',
        ),
        'slicewp_commission_currency_sign' => array(
            'label'     => __( 'Commission currency sign', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => '$',
        ),
        'slicewp_commission_amount_formatted' => array(
            'label'     => __( 'Commission amount formatted', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => '$123.45',
        ),
        'slicewp_commission_reference' => array(
            'label'     => __( 'Commission reference', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => '0123',
        ),
    );

    /**
     * Filter commission tags
     *
     * @since 1.0.0
     *
     * @param array $tags
     *
     * @return array
     */
    return apply_filters( 'automatorwp_slicewp_commission_tags', $commission_tags );

}

/**
 * Custom trigger tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_slicewp_get_trigger_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {

    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Bail if no order ID attached
    if( ! $trigger_args ) {
        return $replacement;
    }

    // Bail if trigger is not from this integration
    if( $trigger_args['integration'] !== 'slicewp' ) {
        return $replacement;
    }

    $tags = array_keys( automatorwp_slicewp_commission_tags() );

    // Bail if not order tags found
    if( ! in_array( $tag_name, $tags ) ) {
        return $replacement;
    }

    $commission_id = (int) automatorwp_get_log_meta( $log->id, 'commission_id', true );

    // Bail if no commission ID attached
    if( $commission_id === 0 ) {
        return $replacement;
    }

    $commission = slicewp_get_commission( $commission_id );

    // Bail if commission can't be found
    if( ! $commission ) {
        return $replacement;
    }

    // Format values for some tags
    switch( $tag_name ) {
        case 'slicewp_commission_id':
            $replacement = $commission->get( 'id' );
            break;
        case 'slicewp_commission_amount':
            $replacement = $commission->get( 'amount' );
            break;
        case 'slicewp_commission_currency':
            $replacement = $commission->get( 'currency' );
            break;
        case 'slicewp_commission_currency_sign':
            $replacement = slicewp_get_currency_symbol( $commission->get( 'currency' ) );
            break;
        case 'slicewp_commission_amount_formatted':
            $replacement = slicewp_format_amount( $commission->get( 'amount' ), $commission->get( 'currency' ) );
            break;
        case 'slicewp_commission_reference':
            $replacement = $commission->get( 'reference' );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_slicewp_get_trigger_tag_replacement', 10, 6 );