<?php
/**
 * Level Added
 *
 * @package     AutomatorWP\Integrations\Restrict_Content\Triggers\Level_Added
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Restrict_Content_Level_Added extends AutomatorWP_Integration_Trigger {

    public $integration = 'restrict_content';
    public $trigger = 'restrict_content_level_added';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User added to access level', 'automatorwp-pro' ),
            'select_option'     => __( 'User added to <strong>access level</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Level. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User added to %1$s %2$s time(s)', 'automatorwp-pro' ), '{access_level}', '{times}' ),
            /* translators: %1$s: Level. */
            'log_label'         => sprintf( __( 'User added to %1$s', 'automatorwp-pro' ), '{access_level}' ),
            'action'            => 'automatorwp_restrict_content_user_added_to_access_level',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'access_level' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'access_level',
                    'name'              => __( 'Access Level:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any access level', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_restrict_content_get_access_levels',
                    'options_cb'        => 'automatorwp_restrict_content_options_cb_access_level',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_restrict_content_get_access_level_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int   $access_level_id
     * @param int       $user_id
     * @param object    $access_level_user
     */
    public function listener( $access_level_id, $user_id, $access_level_user ) {

        // Bail if no logged
        if ( $user_id === 0 ) {
            return;
        }

        // Trigger the level added
        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user_id,
            'access_level_id'           => $access_level_id,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['access_level_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( $trigger_options['access_level'] !== 'any' && absint( $trigger_options['access_level'] ) !== absint( $event['access_level_id'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['access_level_id'] = ( isset( $event['access_level_id'] ) ? $event['access_level_id'] : '' );

        return $log_meta;

    }

}

new AutomatorWP_Restrict_Content_Level_Added();