<?php
/**
 * Anonymous Watch Video Percent
 *
 * @package     AutomatorWP\Integrations\Presto_Player\Triggers\Anonymous_Watch_Video_Percent
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Presto_Player_Anonymous_Watch_Video_Percent extends AutomatorWP_Integration_Trigger {

    public $integration = 'presto_player';
    public $trigger = 'presto_player_anonymous_watch_video_percent';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            'label'             => __( 'Guest watches a percent of a video greater than, less than or equal to a specific percent', 'automatorwp-pro' ),
            'select_option'     => __( 'Guest watches a percent of a video <strong>greater than, less than or equal</strong> to a specific percent', 'automatorwp-pro' ),
            /* translators: %1$s: Post Title. %2$s: Condition. %3$s: Percent. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'Guest watches a percent of %1$s %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{post}', '{condition}', '{percent}', '{times}' ),
            /* translators: %1$s: Post Title. %2$s: Condition. %3$s: Percent. */
            'log_label'         => sprintf( __( 'Guest watches a percent of %1$s %2$s %3$s', 'automatorwp-pro' ), '{post}', '{condition}', '{percent}' ),
            'action'            => 'presto_player_progress',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Video:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any video', 'automatorwp-pro' ),
                    'post_type' => 'pp_video_block'
                ) ),
                'condition' => automatorwp_utilities_condition_option(),
                'percent' => array(
                    'from' => 'percent',
                    'fields' => array(
                        'percent' => array(
                            'name' => __( 'Percent:', 'automatorwp-pro' ),
                            'desc' => __( 'The percent to watch required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                                'step' => '1',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_presto_player_video_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $video_id
     * @param int $percent
     */
    public function listener( $video_id, $percent ) {

        $user_id = get_current_user_id();

        // Bail if user is logged in
        if( $user_id !== 0 ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'video_id'          => $video_id,
            'percent'           => $percent,
        ) );

    }

    /**
     * Anonymous deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function anonymous_deserves_trigger( $deserves_trigger, $trigger, $event, $trigger_options, $automation ) {

        // Don't deserve if video is not received
        if( ! isset( $event['video_id'] ) ) {
            return false;
        }

        $video_id = absint( $event['video_id'] );

        // Don't deserve if video doesn't exists
        if( $video_id === 0 ) {
            return false;
        }

        $model = new \PrestoPlayer\Models\Video();

        if( ! $model ) {
            return '';
        }

        $video = $model->get( $video_id );
        
        $video_post_id = $video->__get( 'post_id' );

        // Don't deserve if video doesn't match with the trigger option
        if( $trigger_options['post'] !== 'any' && absint( $video_post_id ) !== absint( $trigger_options['post'] ) ) {
            return false;
        }

        $percent = $event['percent'];
        $required_percent =  (float) $trigger_options['percent'];

        // Don't deserve if percent watched doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $percent, $required_percent, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_anonymous_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['video_id'] = ( isset( $event['video_id'] ) ? $event['video_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_Presto_Player_Anonymous_Watch_Video_Percent();