<?php
/**
 * Functions
 *
 * @package     AutomatorWP\OpenAI\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Helper function to get the OpenAI url
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_openai_get_url() {

    return 'https://api.openai.com';

}

/**
 * Helper function to get the OpenAI API parameters
 *
 * @since 1.0.0
 *
 * @return array|false
 */
function automatorwp_openai_get_api() {

    $url = automatorwp_openai_get_url();
    $token = automatorwp_openai_get_option( 'token', '' );
    $model = automatorwp_openai_get_option( 'model', '' );

    if( empty( $token ) ) {
        return false;
    }

    return array(
        'url' => $url,
        'token' => $token,
        'model' => $model,
    );

}


/**
 * Get OpenAI chat completions response
 *
 * @since 1.0.0
 * 
 * @param array     $payload     The payload
 */
function automatorwp_openai_chat_completions_response( $payload ) {

    $api = automatorwp_openai_get_api();

    if( ! $api ) {
        return;
    }

    $response = wp_remote_post( $api['url'] . '/v1/chat/completions', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api['token'],
            'Accept' => 'application/json',
            'Content-Type'  => 'application/json'
        ),
        'body' => json_encode( array(
            'model'     => $api['model'],
            'messages'    => array( array(
                'role'      => 'user',
                'content' => $payload['prompt'],
            ) ),
        ) ),
        'timeout' => 60
    ) );

    if ( is_wp_error($response) || $response['response']['code'] !== 200 ) {
        $response = json_decode( wp_remote_retrieve_body( $response ), true  );
        return $response;
    }

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
    
    // to eliminate quotes at the start and end of the response
    $content = trim( $response['choices'][0]['message']['content'], "\"\'" );
    
    return $content;
}

/**
 * Get OpenAI image generation response
 *
 * @since 1.0.0
 * 
 * @param array     $payload     The payload
 */
function automatorwp_openai_image_generation_response( $payload ) {

    $api = automatorwp_openai_get_api();

    if( ! $api ) {
        return;
    }

    $response = wp_remote_post( $api['url'] . '/v1/images/generations', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api['token'],
            'Accept' => 'application/json',
            'Content-Type'  => 'application/json'
        ),
        'body' => json_encode( array(
            'model'     => $payload['model'],
            'prompt'    => $payload['prompt'],
            'n'         => 1,
            'size'      => $payload['size'],
            ),
        ),
        'timeout' => 60
    ) );

    if ( is_wp_error($response) || $response['response']['code'] !== 200 ) {
        return false;
    }

    $response = json_decode( wp_remote_retrieve_body( $response ), true  );
    
    return $response['data'][0]['url'];
}

/**
 * Get languages
 *
 * @since 1.0.0
 * 
 * @return array     $options     The languages
 */
function automatorwp_openai_get_languages() {

    $options = array(
        'English'    => __( 'English', 'automatorwp-pro' ),
        'Spanish'    => __( 'Spanish', 'automatorwp-pro' ),
        'German'     => __( 'German', 'automatorwp-pro' ),
        'French'     => __( 'French', 'automatorwp-pro' ),
        'Italian'    => __( 'Italian', 'automatorwp-pro' ),
        'Russian'    => __( 'Russian', 'automatorwp-pro' ),
        'Dutch'      => __( 'Dutch', 'automatorwp-pro' ),
        'Swedish'    => __( 'Swedish', 'automatorwp-pro' ),
        'Polish'     => __( 'Polish', 'automatorwp-pro' ),
        'Romanian'   => __( 'Romanian', 'automatorwp-pro' ),
        'Greek'      => __( 'Greek', 'automatorwp-pro' ),
        'Danish'     => __( 'Danish', 'automatorwp-pro' ),
        'Hungarian'  => __( 'Hungarian', 'automatorwp-pro' ),
        'Czech'      => __( 'Czech', 'automatorwp-pro' ),
        'Chinese'    => __( 'Chinese', 'automatorwp-pro' ),
        'Japanese'   => __( 'Japanese', 'automatorwp-pro' ),
        'Korean'     => __( 'Korean', 'automatorwp-pro' ),
        'Vietnamese' => __( 'Vietnamese', 'automatorwp-pro' ),
        'Indonesian' => __( 'Indonesian', 'automatorwp-pro' ),
        'Thai'       => __( 'Thai', 'automatorwp-pro' ),
        'Turkish'    => __( 'Turkish', 'automatorwp-pro' ),
        'Arabic'     => __( 'Arabic', 'automatorwp-pro' ),
        'Hebrew'     => __( 'Hebrew', 'automatorwp-pro' ),
    );
    
    return $options;

}

/**
 * Image size
 *
 * @since 1.0.0
 * 
 * @return array     $options     Image size
 */
function automatorwp_openai_get_size() {

    $options = array(
        '256x256'   => '256x256',
        '512x512'   => '512x512',
        '1024x1024' => '1024x1024',
    );
    
    return $options;

}