<?php
/**
 * Generate SEO title
 *
 * @package     AutomatorWP\Integrations\OpenAI\Actions\Generate_SEO_Title
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_OpenAI_Generate_SEO_Title extends AutomatorWP_Integration_Action {

    public $integration = 'openai';
    public $action = 'openai_generate_seo_title';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Generate a SEO title', 'automatorwp-pro' ),
            'select_option'     => __( 'Generate a SEO <strong>title</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Prompt. */
            'edit_label'        => sprintf( __( 'Generate a SEO %1$s title', 'automatorwp-pro' ), '{prompt}' ),
            /* translators: %1$s: Prompt. */
            'log_label'         => sprintf( __( 'Generate a SEO %1$s title', 'automatorwp-pro' ), '{prompt}' ),
            'options'           => array(
                'prompt' => array(
                    'from' => 'prompt',
                    'default' => __( 'title', 'automatorwp-pro' ),
                    'fields' => array(
                        'prompt' => array(
                            'name' => __( 'Title:', 'automatorwp-pro' ),
                            'desc' => __( 'The SEO title topic.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                    ),
            ),
            'tags' => automatorwp_openai_get_actions_response_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $default_prompt =  __( 'Generate an SEO-friendly title for the following content that is no more than 55 characters long (and must never exceed 60 characters, including spaces and special characters', 'automatorwp-pro' );
        
        // Bail if prompt is empty
        if ( empty( $action_options['prompt'] ) ) {
            $this->result = __( 'Prompt field is empty.', 'automatorwp-pro' );
            return;
        }

        $payload = array(
            'prompt'    => $default_prompt . ': ' . $action_options['prompt'],
        );

        $this->result = '';
        $this->response = '';
        
        $api = automatorwp_openai_get_api();

        if( ! $api ) {
            return $options;
        }

        $this->response = automatorwp_openai_chat_completions_response( $payload );

        if ( ! $this->response || isset( $this->response['error'] ) ) {
			
			if ( isset( $this->response['error']['message'] ) ){
				$this->result = sprintf( __( '%s', 'automatorwp-pro' ), $this->response['error']['message'] );
			} else {
				$this->result = __( 'Error: Please check your OpenAI configuration.', 'automatorwp-pro' );	
			}
			$this->response = '';
            return;
        }
    
        $this->result = sprintf( __( '%s', 'automatorwp-pro' ), $this->response );

        }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Warn user if the authorization has not been setup from settings
        if( ! automatorwp_openai_get_api() ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">OpenAI settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-openai'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/openai/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['response'] = ( isset( $this->response ) ? $this->response : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_OpenAI_Generate_SEO_Title();