<?php
/**
 * Book User
 *
 * @package     AutomatorWP\Integrations\Modern_Events_Calendar\Actions\Book_User
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Modern_Events_Calendar_Book_User extends AutomatorWP_Integration_Action {

    public $integration = 'modern_events_calendar';
    public $action = 'modern_events_calendar_book_user';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Register user for an event', 'automatorwp-pro' ),
            'select_option'     => __( 'Register user for an <strong>event</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: User. %2$s: Post title. */
            'edit_label'        => sprintf( __( 'Register %1$s for %2$s', 'automatorwp-pro' ), '{user}', '{event}' ),
            /* translators: %1$s: User. %2$s: Post title. */
            'log_label'         => sprintf( __( 'Register %1$s for %2$s', 'automatorwp-pro' ), '{user}', '{event}'  ),
            'options'           => array(
                'user' => array(
                    'from' => 'user',
                    'default' => __( 'user', 'automatorwp-pro' ),
                    'fields' => array(
                        'user' => array(
                            'name' => __( 'User ID:', 'automatorwp-pro' ),
                            'desc' => __( 'User ID to register. Leave blank to register the user that completes the automation.', 'automatorwp-pro' ),
                            'type' => 'input',
                            'default' => ''
                        ),
                    )
                ),
                'event' => array(
                    'from' => 'event',
                    'option_default' => __( 'an event', 'automatorwp-pro' ),
                    'fields' => array(
                        'event' => automatorwp_utilities_post_field( array(
                            'name'                  => __( 'Event:', 'automatorwp-pro' ),
                            'option_none'           => true,
                            'option_none_label'     => __( 'an event', 'automatorwp-pro' ),
                            'option_custom'         => false,
                            'post_type'             => 'mec-events',
                            'placeholder'           => __( 'Select an event', 'automatorwp-pro' ),
                            'default'               => 'any'
                        ) ),
                        'ticket' => automatorwp_utilities_ajax_selector_field( array(
                            'name'              => __( 'Ticket:', 'automatorwp-pro' ),
                            'option_none'       => false,
                            'action_cb'         => 'automatorwp_modern_events_calendar_get_event_tickets',
                            'options_cb'        => 'automatorwp_modern_events_calendar_options_cb_event_tickets',
                            'placeholder'       => __( 'Select a ticket', 'automatorwp-pro' ),
                            'default'           => '',
                        ) ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $event_id = absint( $action_options['event'] );
        $ticket_id = absint( $action_options['ticket'] );
        $user_id_to_register = absint( $action_options['user'] );

        // Bail if MEC_gateway_pay_locally class not exists
        if ( ! class_exists( '\MEC_gateway_pay_locally' ) ) {
            return;
        }

        // Bail if MEC_feature_books class not exists
        if ( ! class_exists( '\MEC_feature_books' ) ) {
            return;
        }

        if( $user_id_to_register === 0 ) {
            $user_id_to_register = $user_id;
        }

        // Bail if not event ID provided
        if( $event_id === 0 ) {
            return;
        }

        $user = get_user_by( 'ID', $user_id_to_register );

        // Bail if user not exists
        if( ! $user ) {
            return;
        }

        $gateway  = new \MEC_gateway_pay_locally();
        $books = new \MEC_feature_books();
        $book = $books->getBook();
        $attendee = array(
            'email' => $user->user_email,
            'name'  => $user->display_name,
            'reg'   => array(),
        );

        // Generate a new user id from the gateway registration
        $user_id = $gateway->register_user( $attendee );

        // Check the event date
        $event_date = get_post_meta( $event_id, 'mec_date', true );

        // Bail if date not properly configured
        if ( ! isset( $event_date['start'] ) || ! isset( $event_date['end'] ) ) {
            return;
        }

        $date = $book->timestamp( $event_date['start'], $event_date['end'] );

        // Setup the tickets
        $tickets = array();

        $tickets[] = array_merge(
            $attendee,
            array(
                'id'         => $ticket_id,
                'count'      => 1,
                'variations' => array(),
                'reg'        => ( isset( $attendee['reg'] ) ? $attendee['reg'] : array() ),
            )
        );

        // Calculate price of bookings
        $event_tickets = get_post_meta( $event_id, 'mec_tickets', true );

        $price_details = $book->get_price_details( array( $ticket_id => 1 ), $event_id, $event_tickets, array() );

        // Add a new transaction
        $transaction_id = $book->temporary( array(
            'tickets'       => $tickets,
            'date'          => $date,
            'event_id'      => $event_id,
            'price_details' => $price_details,
            'total'         => $price_details['total'],
            'discount'      => 0,
            'price'         => $price_details['total'],
            'coupon'        => null,
            'fields'        => array(),
        ) );

        // Add a new booking
        $book_args = array(
            'post_title'    => $user->display_name . ' - ' . $user->user_email,
            'post_type'     => 'mec-books',
            'post_author'   => $user_id,
        );

        $booking_id = $book->add( $book_args, $transaction_id, ',' . $ticket_id . ',' );

        // Update the booking metas
        update_post_meta( $booking_id, 'mec_attendees', $tickets );
        update_post_meta( $booking_id, 'mec_reg', ( isset( $attendee['reg'] ) ? $attendee['reg'] : array() ) );
        update_post_meta( $booking_id, 'mec_gateway', 'MEC_gateway_pay_locally' );
        update_post_meta( $booking_id, 'mec_gateway_label', $gateway->title() );
        update_post_meta( $booking_id, 'mec_book_date_submit', gmdate( 'YmdHis', time() ) );

        // Fire pending booking action
        do_action( 'mec_booking_pended', $booking_id );

    }

}

new AutomatorWP_Modern_Events_Calendar_Book_User();