<?php
/**
 * Add User Segment
 *
 * @package     AutomatorWP\Integrations\Mautic\Actions\Add_User_Segment
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Mautic_Add_User_Segment extends AutomatorWP_Integration_Action {

    public $integration = 'mautic';
    public $action = 'mautic_add_user_segment';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add user to segment', 'automatorwp-pro' ),
            'select_option'     => __( 'Add user to <strong>segment</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Segment title. */
            'edit_label'        => sprintf( __( 'Add user to %1$s', 'automatorwp-pro' ), '{segment}' ),
            /* translators: %1$s: Segment title. */
            'log_label'         => sprintf( __( 'Add user to %1$s', 'automatorwp-pro' ), '{segment}' ),
            'options'           => array(
                'segment' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'segment',
                    'name'              => __( 'Segment:', 'automatorwp-pro' ),
                    'option_default'    => __( 'Select a segment', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_mautic_get_segments',
                    'options_cb'        => 'automatorwp_mautic_options_cb_segment',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Bail if Mautic not configured
        $params = automatorwp_mautic_get_request_parameters( );

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'Could not add contact to segment, Mautic authentication failed', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $user = get_userdata( $user_id );
        $segment_id = $action_options['segment'];
        $contact_id = automatorwp_mautic_get_contact_id ( $user->user_email );

        // Bail if empty segment
        if ( empty ( $segment_id ) ) {
            $this->result = __( 'No segment selected.', 'automatorwp-pro' );
            return;
        }

        if ( empty ( $contact_id ) ) {
            $this->result = sprintf( __( '%s is not registered in Mautic.', 'automatorwp-pro' ), $user->user_email );
            return;
        }

        $contact_segments = automatorwp_mautic_get_contact_segments ( $contact_id );

        // Check if user belongs to segments
        if ( ! empty ( $contact_segments ) ) {

            if ( in_array ( $segment_id, $contact_segments ) ) {
                $this->result = sprintf( __( '%s is already a member of the segment.', 'automatorwp-pro' ), $user->user_email );
                return;
            }

        }

        $code = automatorwp_mautic_add_contact_segment( $segment_id, $contact_id );
        
        if ( $code !== 200 ) {
            $this->result = sprintf( __( 'Error %d. %s not added to segment', 'automatorwp-pro' ), $code, $user->user_email );
        } else {
            $this->result = sprintf( __( '%s added to segment', 'automatorwp-pro' ), $user->user_email );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        $params = automatorwp_mautic_get_request_parameters( );

        // Warn user if the authorization has not been setup from settings
        if( $params === false ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Mautic settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-mautic'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/mautic/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Mautic_Add_User_Segment();