<?php
/**
 * Functions
 *
 * @package     AutomatorWP\LearnDash\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Retrieves the post ID.
 *
 * @since 1.0.0
 *
 * @param mixed $thing
 *
 * @return int|false
 */
function automatorwp_learndash_get_post_id( $thing ) {

    if( $thing instanceof WP_Post ) {
        return absint( $thing->ID );
    }

    if( is_numeric( $thing ) ) {

        if( absint( $thing ) === 0 ) {
            return false;
        } else {
            return absint( $thing );
        }
    }

    return false;
}

/**
 * Helper function to mark a quiz as completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $quiz_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_quiz_as_completed( $user_id = 0, $quiz_id = 0, $course_id = 0 ) {

    $user_meta      = get_user_meta( $user_id, '_sfwd-quizzes', true );
    $quiz_progress  = empty( $user_meta ) ? array() : $user_meta;

    $quiz_meta = get_post_meta( $quiz_id, '_sfwd-quiz', true );

    $quiz_data = array(
        'quiz'             => $quiz_id,
        'score'            => 0,
        'count'            => 0,
        'pass'             => true,
        'rank'             => '-',
        'time'             => time(),
        'pro_quizid'       => $quiz_meta['sfwd-quiz_quiz_pro'],
        'course'           => $course_id,
        'points'           => 0,
        'total_points'     => 0,
        'percentage'       => 0,
        'timespent'        => 0,
        'has_graded'       => false,
        'statistic_ref_id' => 0,
        'm_edit_by'        => 9999999,  // Manual Edit By ID.
        'm_edit_time'      => time(),   // Manual Edit timestamp.
    );

    $quiz_progress[] = $quiz_data;

    // Add the quiz entry to the user activity
    learndash_update_user_activity(
        array(
            'course_id'          => $course_id,
            'user_id'            => $user_id,
            'post_id'            => $quiz_id,
            'activity_type'      => 'quiz',
            'activity_action'    => 'insert',
            'activity_status'    => $quiz_data['pass'],
            'activity_started'   => $quiz_data['time'],
            'activity_completed' => $quiz_data['time'],
            'activity_meta'      => $quiz_data,
        )
    );

    // Update user quiz progress
    if ( ! empty( $quiz_progress ) ) {
        update_user_meta( $user_id, '_sfwd-quizzes', $quiz_progress );
    }

}

/**
 * Helper function to mark a topic as completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $topic_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_topic_as_completed( $user_id = 0, $topic_id = 0, $course_id = 0 ) {

    // Get all topic's quizzes
    $quizzes = learndash_get_lesson_quiz_list( $topic_id, $user_id, $course_id ); // learndash_get_lesson_quiz_list() works for topics too

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Mark quiz as completed
            automatorwp_learndash_mark_quiz_as_completed( $user_id, $quiz_data['post']->ID, $course_id );
        }

    }

    // Mark topic as completed
    learndash_process_mark_complete( $user_id, $topic_id, false, $course_id );

}

/**
 * Helper function to mark a topic as not completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $topic_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_topic_as_not_completed( $user_id = 0, $topic_id = 0, $course_id = 0 ) {

    // Get all topic quizzes
    $quizzes = learndash_get_lesson_quiz_list( $topic_id, $user_id, $course_id ); // learndash_get_lesson_quiz_list() works for topics too

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Reset quiz progress
            learndash_delete_quiz_progress( $user_id, $quiz_data['post']->ID );
        }

    }

    // Mark topic as not completed
    learndash_process_mark_incomplete( $user_id, $course_id, $topic_id, false );

}

/**
 * Helper function to mark a lesson as completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $lesson_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_lesson_as_completed( $user_id = 0, $lesson_id = 0, $course_id = 0 ) {

    // Get all lesson topics
    $topics = learndash_get_topic_list( $lesson_id, $course_id );

    if( is_array( $topics ) ) {

        foreach( $topics as $topic ) {
            // Mark topic as completed
            automatorwp_learndash_mark_topic_as_completed( $user_id, $topic->ID, $course_id );
        }

    }

    // Get all lesson's quizzes
    $quizzes = learndash_get_lesson_quiz_list( $lesson_id, $user_id, $course_id );

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Mark quiz as completed
            automatorwp_learndash_mark_quiz_as_completed( $user_id, $quiz_data['post']->ID, $course_id );
        }

    }

    // Mark lesson as completed
    learndash_process_mark_complete( $user_id, $lesson_id, false, $course_id );

}

/**
 * Helper function to mark a lesson as not completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $lesson_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_lesson_as_not_completed( $user_id = 0, $lesson_id = 0, $course_id = 0 ) {

    // Get all lesson topics
    $topics = learndash_get_topic_list( $lesson_id, $course_id );

    if( is_array( $topics ) ) {

        foreach( $topics as $topic ) {
            // Mark topic as not completed
            automatorwp_learndash_mark_topic_as_not_completed( $user_id, $topic->ID, $course_id );
        }

    }

    // Get all lesson quizzes
    $quizzes = learndash_get_lesson_quiz_list( $lesson_id, $user_id, $course_id );

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Reset quiz progress
            learndash_delete_quiz_progress( $user_id, $quiz_data['post']->ID );
        }

    }

    // Mark lesson as not completed
    learndash_process_mark_incomplete( $user_id, $course_id, $lesson_id, false );

}

/**
 * Helper function to mark a course as completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_course_as_completed( $user_id = 0, $course_id = 0 ) {

    // Get all course lessons
    $lessons = learndash_get_lesson_list( $course_id, array( 'num' => 0 ) );

    if( is_array( $lessons ) ) {

        foreach( $lessons as $lesson ) {
            // Mark lesson as completed
            automatorwp_learndash_mark_lesson_as_completed( $user_id, $lesson->ID, $course_id );
        }

    }

    // Get all course quizzes
    $quizzes = learndash_get_course_quiz_list( $course_id, $user_id );

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Mark course quizzes as completed
            automatorwp_learndash_mark_quiz_as_completed( $user_id, $quiz_data['post']->ID, $course_id );
        }

    }

    // Mark course as completed
    learndash_process_mark_complete( $user_id, $course_id, false, $course_id );

}

/**
 * Helper function to mark a course as not completed
 *
 * @since 1.0.0
 *
 * @param int $user_id
 * @param int $course_id
 * @param int $course_id
 */
function automatorwp_learndash_mark_course_as_not_completed( $user_id = 0, $course_id = 0 ) {

    // Get all course lessons
    $lessons = learndash_get_lesson_list( $course_id, array( 'num' => 0 ) );

    if( is_array( $lessons ) ) {

        foreach( $lessons as $lesson ) {
            // Mark lesson as not completed
            automatorwp_learndash_mark_lesson_as_not_completed( $user_id, $lesson->ID, $course_id );
        }

    }

    // Get all course quizzes
    $quizzes = learndash_get_course_quiz_list( $course_id, $user_id );

    if( is_array( $quizzes ) ) {

        foreach( $quizzes as $quiz_data ) {
            // Delete course quizzes progress
            learndash_delete_quiz_progress( $user_id, $quiz_data['post']->ID );
        }

    }

    // Mark course as not completed
    learndash_delete_course_progress( $course_id, $user_id );

}

/**
 * Helper function to generate certificates
 *
 * @since 1.0.0
 * 
 * @param        $args
 * @param        $body
 * @param string $certificate_type
 *
 * @return array
 */
function automatorwp_learndash_generate_certificate( $args, $body, $course_id, $certificate_type = 'course' ) {

	$save_path = $args['save_path'];
	$file_name = $args['file_name'];
	$user = ( isset( $args['user'] ) ) ? $args['user'] : wp_get_current_user();

	$cert_args_defaults = array(
		'cert_id'       => 0,   
		'post_id'       => 0,
		'user_id'       => 0,
		'lang'          => 'eng',
		'filename'      => '',
		'filename_url'  => '',
		'filename_type' => 'title',
		'pdf_title'     => '',
		'ratio'         => 1.25,
	);

	$cert_args = shortcode_atts( $cert_args_defaults, $args );

	// Just to ensure we have valid IDs.
	$cert_args['cert_id'] = absint( $args['certificate_post'] );
	$cert_args['user_id'] = absint( $user->ID );

	if ( 'preview' === (string) $certificate_type || 'automatorwp' === (string) $certificate_type ) {
		$cert_args['post_id'] = absint( $args['certificate_post'] );
	}

	if ( empty( $cert_args['cert_id'] ) ) {
		if ( isset( $_GET['id'] ) ) {
			$cert_args['cert_id'] = absint( $_GET['id'] );
		} else {
			$cert_args['cert_id'] = get_the_id();
		}
	}

	if ( empty( $cert_args['user_id'] ) ) {
		if ( isset( $_GET['user'] ) ) {
			$cert_args['user_id'] = absint( $_GET['user'] );
		} elseif ( isset( $_GET['user_id'] ) ) {
			$cert_args['user_id'] = absint( $_GET['user_id'] );
		}
	}

	$cert_args['cert_post'] = get_post( $cert_args['cert_id'] );

	if ( ( ! $cert_args['cert_post'] ) || ( ! is_a( $cert_args['cert_post'], 'WP_Post' ) ) || ( learndash_get_post_type_slug( 'certificate' ) !== $cert_args['cert_post']->post_type ) ) {
		return false;
	}

	$cert_args['post_post'] = get_post( $cert_args['post_id'] );

	if ( ( ! $cert_args['post_post'] ) || ( ! is_a( $cert_args['post_post'], 'WP_Post' ) ) ) {
		return false;
	}

	$cert_args['user'] = get_user_by( 'ID', $cert_args['user_id'] );

	if ( ( ! $cert_args['user'] ) || ( ! is_a( $cert_args['user'], 'WP_User' ) ) ) {
		return false;
	}

	$config_lang_tmp = 'eng';

	if ( ( isset( $_GET['cert_lang'] ) ) && ( ! empty( $_GET['cert_lang'] ) ) ) {
		$config_lang_tmp = substr( esc_attr( $_GET['cert_lang'] ), 0, 3 );
	} elseif ( ( isset( $_GET['lang'] ) ) && ( ! empty( $_GET['lang'] ) ) ) {
		$config_lang_tmp = substr( esc_attr( $_GET['lang'] ), 0, 3 );
	}

	if ( ( ! empty( $config_lang_tmp ) ) && ( strlen( $config_lang_tmp ) == 3 ) ) {
		$cert_lang_dir = LEARNDASH_LMS_LIBRARY_DIR . '/tcpdf/config/lang';
		$lang_files = array_diff(
			scandir( $cert_lang_dir ),
			array(
				'..',
				'.',
			)
		);
		if ( ( ! empty( $lang_files ) ) && ( is_array( $lang_files ) ) && ( in_array( $config_lang_tmp, $lang_files, true ) ) && ( file_exists( $cert_lang_dir . '/' . $config_lang_tmp . '.php' ) ) ) {
			$cert_args['lang'] = $config_lang_tmp;
		}
	}

	$target_post_id = 0;
	$cert_args['filename_type'] = 'title';
	$logo_file = '';
	$logo_enable = '';
	$subsetting_enable = '';
	$filters = '';
	$header_enable = '';
	$footer_enable = '';
	$monospaced_font = '';
	$font = '';
	$font_size = '';

	ob_start();

	$cert_args['cert_title'] = $cert_args['cert_post']->post_title;
	$cert_args['cert_title'] = strip_tags( $cert_args['cert_title'] );

	/** This filter is documented in https://developer.wordpress.org/reference/hooks/document_title_separator/ */
	$sep = apply_filters( 'document_title_separator', '-' );

	/**
	 * Filters username of the user to be used in creating certificate PDF.
	 *
	 * @param string $user_name User display name.
	 * @param int $user_id User ID.
	 * @param int $cert_id Certificate post ID.
	 */
	$learndash_pdf_username = apply_filters( 'learndash_pdf_username', $cert_args['user']->display_name, $cert_args['user_id'], $cert_args['cert_id'] );
	if ( ! empty( $learndash_pdf_username ) ) {
		if ( ! empty( $cert_args['pdf_title'] ) ) {
			$cert_args['pdf_title'] .= " $sep ";
		}
		$cert_args['pdf_title'] .= $learndash_pdf_username;
	}

	$cert_for_post_title = get_the_title( $cert_args['post_id'] );
	strip_tags( $cert_for_post_title );
	if ( ! empty( $cert_for_post_title ) ) {
		if ( ! empty( $cert_args['pdf_title'] ) ) {
			$cert_args['pdf_title'] .= " $sep ";
		}
		$cert_args['pdf_title'] .= $cert_for_post_title;
	}

	if ( ! empty( $cert_args['pdf_title'] ) ) {
		$cert_args['pdf_title'] .= " $sep ";
	}
	$cert_args['pdf_title'] .= $cert_args['cert_title'];

	if ( ! empty( $cert_args['pdf_title'] ) ) {
		$cert_args['pdf_title'] .= " $sep ";
	}
	$cert_args['pdf_title'] .= get_bloginfo( 'name', 'display' );

	$cert_args['cert_permalink']  = get_permalink( $cert_args['cert_post']->ID );
	$cert_args['pdf_author_name'] = $cert_args['user']->display_name;

	$tags_array                = array();
	$cert_args['pdf_keywords'] = '';
	$tags_data                 = wp_get_post_tags( $cert_args['cert_post']->ID );

	if ( $tags_data ) {
		foreach ( $tags_data as $val ) {
			$tags_array[] = $val->name;
		}
		$cert_args['pdf_keywords'] = implode( ' ', $tags_array );
	}

	if ( ! empty( $_GET['font'] ) ) {
		$font = esc_html( $_GET['font'] );
	}

	if ( ! empty( $_GET['monospaced'] ) ) {
		$monospaced_font = esc_html( $_GET['monospaced'] );
	}

	if ( ! empty( $_GET['fontsize'] ) ) {
		$font_size = intval( $_GET['fontsize'] );
	}

	if ( ! empty( $_GET['subsetting'] ) && ( $_GET['subsetting'] == 1 || $_GET['subsetting'] == 0 ) ) {
		$subsetting_enable = $_GET['subsetting'];
	}

	if ( $subsetting_enable == 1 ) {
		$subsetting = 'true';
	} else {
		$subsetting = 'false';
	}

	if ( ! empty( $_GET['ratio'] ) ) {
		$cert_args['ratio'] = floatval( $_GET['ratio'] );
	}

	if ( ! empty( $_GET['header'] ) ) {
		$header_enable = $_GET['header'];
	}

	if ( ! empty( $_GET['logo'] ) ) {
		$logo_enable = $_GET['logo'];
	}

	if ( ! empty( $_GET['logo_file'] ) ) {
		$logo_file = esc_html( $_GET['logo_file'] );
	}

	if ( ! empty( $_GET['logo_width'] ) ) {
		$logo_width = intval( $_GET['logo_width'] );
	}

	if ( ! empty( $_GET['footer'] ) ) {
		$footer_enable = $_GET['footer'];
	}

	/**
	 * Start Cert post content processing.
	 */
	if ( ! defined( 'LEARNDASH_TCPDF_LEGACY_LD322' ) ) {
		$use_LD322_define = apply_filters( 'learndash_tcpdf_legacy_ld322', true, $cert_args );
		define( 'LEARNDASH_TCPDF_LEGACY_LD322', $use_LD322_define );
	}

	$cert_content = ! empty( $body ) ? html_entity_decode( $body ) : $cert_args['cert_post']->post_content;

	// Delete shortcode for POST2PDF Converter
	$cert_content = preg_replace( '|\[pdf[^\]]*?\].*?\[/pdf\]|i', '', $cert_content );

	$cert_content = automatorwp_learndash_certificate_content( $cert_content, $args, $course_id );

	$cert_content = do_shortcode( $cert_content );

	// Convert relative image path to absolute image path
	$cert_content = preg_replace( "/<img([^>]*?)src=['\"]((?!(http:\/\/|https:\/\/|\/))[^'\"]+?)['\"]([^>]*?)>/i", '<img$1src="' . site_url() . '/$2"$4>', $cert_content );

	// Set image align to center
	$cert_content = preg_replace_callback(
		"/(<img[^>]*?class=['\"][^'\"]*?aligncenter[^'\"]*?['\"][^>]*?>)/i",
		//array(
			
			'learndash_post2pdf_conv_image_align_center',
		//),
		$cert_content
	);

	// Add width and height into image tag
	$cert_content = preg_replace_callback(
		"/(<img[^>]*?src=['\"]((http:\/\/|https:\/\/|\/)[^'\"]*?(jpg|jpeg|gif|png))['\"])([^>]*?>)/i",
		//array(
			
			'learndash_post2pdf_conv_img_size',
		//),
		$cert_content
	);

	if ( ( ! defined( 'LEARNDASH_TCPDF_LEGACY_LD322' ) ) || ( true !== LEARNDASH_TCPDF_LEGACY_LD322 ) ) {
		$cert_content = wpautop( $cert_content );
	}

	// For other sourcecode
	$cert_content = preg_replace( '/<pre[^>]*?><code[^>]*?>(.*?)<\/code><\/pre>/is', '<pre style="word-wrap:break-word; color: #406040; background-color: #F1F1F1; border: 1px solid #9F9F9F;">$1</pre>', $cert_content );

	// For blockquote
	$cert_content = preg_replace( '/<blockquote[^>]*?>(.*?)<\/blockquote>/is', '<blockquote style="color: #406040;">$1</blockquote>', $cert_content );

	$cert_content = '<br/><br/>' . $cert_content;

	/**
	 * If the $font variable is not empty we use it to replace all font
	 * definitions. This only affects inline styles within the structure
	 * of the certificate content HTML elements.
	 */
	if ( ! empty( $font ) ) {
		$cert_content = preg_replace( '/(<[^>]*?font-family[^:]*?:)([^;]*?;[^>]*?>)/is', '$1' . $font . ',$2', $cert_content );
	}

	if ( ( defined( 'LEARNDASH_TCPDF_LEGACY_LD322' ) ) && ( true === LEARNDASH_TCPDF_LEGACY_LD322 ) ) {
		$cert_content = preg_replace( '/\n/', '<br/>', $cert_content ); //"\n" should be treated as a next line
	}

	/**
	 * Filters whether to include certificate CSS styles in certificate content or not.
	 *
	 * @param boolean $include_certificate_styles 	Whether to include certificate styles.
	 * @param int 	$cert_id 						Certificate post ID.
	 */
	if ( apply_filters( 'learndash_certificate_styles', true, $cert_args['cert_id'] ) ) {
		$certificate_styles = \LearnDash_Settings_Section::get_section_setting( 'LearnDash_Settings_Certificates_Styles', 'styles' );
		
		$certificate_styles = preg_replace( '/<style[^>]*?>(.*?)<\/style>/is', '$1', $certificate_styles );
		if ( ! empty( $certificate_styles ) ) {
			$cert_content = '<style>' . $certificate_styles . '</style>' . $cert_content;
		}
	}

	/**
	 * Filters certificate content after all processing.
	 *
	 * @param string $cert_content Certificate post content HTML/TEXT.
	 * @param int $cert_id Certificate post ID.
	 *
	 * @since 3.2.0
	 *
	 */
	$cert_content = apply_filters( 'learndash_certificate_content', $cert_content, $cert_args['cert_id'] );

	/**
	 * Build the PDF Certificate using TCPDF.
	 */
	if ( ! class_exists( 'TCPDF' ) ) {
		require_once LEARNDASH_LMS_LIBRARY_DIR . '/tcpdf/config/lang/' . $cert_args['lang'] . '.php';
		require_once LEARNDASH_LMS_LIBRARY_DIR . '/tcpdf/tcpdf.php';
	}

	$learndash_certificate_options = get_post_meta( $cert_args['cert_post']->ID, 'learndash_certificate_options', true );
	if ( ! is_array( $learndash_certificate_options ) ) {
		$learndash_certificate_options = array( $learndash_certificate_options );
	}

	if ( ! isset( $learndash_certificate_options['pdf_page_format'] ) ) {
		$learndash_certificate_options['pdf_page_format'] = PDF_PAGE_FORMAT;
	}

	if ( ! isset( $learndash_certificate_options['pdf_page_orientation'] ) ) {
		$learndash_certificate_options['pdf_page_orientation'] = PDF_PAGE_ORIENTATION;
	}

	// Create a new object
	$tcpdf_params = array(
		'orientation' => $learndash_certificate_options['pdf_page_orientation'],
		'unit'        => PDF_UNIT,
		'format'      => $learndash_certificate_options['pdf_page_format'],
		'unicode'     => true,
		'encoding'    => 'UTF-8',
		'diskcache'   => false,
		'pdfa'        => false,
		'margins'     => array(
			'top'    => PDF_MARGIN_TOP,
			'right'  => PDF_MARGIN_RIGHT,
			'bottom' => PDF_MARGIN_BOTTOM,
			'left'   => PDF_MARGIN_LEFT,
		),
	);

	/**
	 * Filters certificate tcpdf paramaters.
	 *
	 * @param array $tcpdf_params An array of tcpdf parameters.
	 * @param int $cert_id Certificate post ID.
	 *
	 * @since 2.4.7
	 *
	 */
	$tcpdf_params = apply_filters( 'learndash_certificate_params', $tcpdf_params, $cert_args['cert_id'] );

	$pdf = new \TCPDF(
		$tcpdf_params['orientation'],
		$tcpdf_params['unit'],
		$tcpdf_params['format'],
		$tcpdf_params['unicode'],
		$tcpdf_params['encoding'],
		$tcpdf_params['diskcache'],
		$tcpdf_params['pdfa']
	);

	// Added to let external manipulate the $pdf instance.
	/**
	 * Fires after creating certificate `TCPDF` class object.
	 *
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 *
	 * @since 2.4.7
	 *
	 */
	do_action( 'learndash_certification_created', $pdf, $cert_args['cert_id'] );

	// Set document information

	/**
	 * Filters the value of pdf creator.
	 *
	 * @param string $pdf_creator The name of the PDF creator.
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 */
	$pdf->SetCreator( apply_filters( 'learndash_pdf_creator', PDF_CREATOR, $pdf, $cert_args['cert_id'] ) );

	/**
	 * Filters the name of the pdf author.
	 *
	 * @param string $pdf_author_name PDF author name.
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 */
	$pdf->SetAuthor( apply_filters( 'learndash_pdf_author', $cert_args['pdf_author_name'], $pdf, $cert_args['cert_id'] ) );

	/**
	 * Filters the title of the pdf.
	 *
	 * @param string $pdf_title PDF title.
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 */
	$pdf->SetTitle( apply_filters( 'learndash_pdf_title', $cert_args['pdf_title'], $pdf, $cert_args['cert_id'] ) );

	/**
	 * Filters the subject of the pdf.
	 *
	 * @param string $pdf_subject PDF subject
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 */
	$pdf->SetSubject( apply_filters( 'learndash_pdf_subject', strip_tags( get_the_category_list( ',', '', $cert_args['cert_id'] ) ), $pdf, $cert_args['cert_id'] ) );

	/**
	 * Filters the pdf keywords.
	 *
	 * @param string $pdf_keywords PDF keywords.
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $cert_id Certificate post ID.
	 */
	$pdf->SetKeywords( apply_filters( 'learndash_pdf_keywords', $cert_args['pdf_keywords'], $pdf, $cert_args['cert_id'] ) );

	// Set header data
	if ( mb_strlen( $cert_args['cert_title'], 'UTF-8' ) < 42 ) {
		$header_title = $cert_args['cert_title'];
	} else {
		$header_title = mb_substr( $cert_args['cert_title'], 0, 42, 'UTF-8' ) . '...';
	}

	if ( $header_enable == 1 ) {
		if ( $logo_enable == 1 && $logo_file ) {
			$pdf->SetHeaderData( $logo_file, $logo_width, $header_title, 'by ' . $cert_args['pdf_author_name'] . ' - ' . $cert_args['cert_permalink'] );
		} else {
			$pdf->SetHeaderData( '', 0, $header_title, 'by ' . $cert_args['pdf_author_name'] . ' - ' . $cert_args['cert_permalink'] );
		}
	}

	// Set header and footer fonts
	if ( $header_enable == 1 ) {
		$pdf->setHeaderFont( array( $font, '', PDF_FONT_SIZE_MAIN ) );
	}

	if ( $footer_enable == 1 ) {
		$pdf->setFooterFont( array( $font, '', PDF_FONT_SIZE_DATA ) );
	}

	// Remove header/footer
	if ( $header_enable == 0 ) {
		$pdf->setPrintHeader( false );
	}

	if ( $header_enable == 0 ) {
		$pdf->setPrintFooter( false );
	}

	// Set default monospaced font
	$pdf->SetDefaultMonospacedFont( $monospaced_font );

	// Set margins
	$pdf->SetMargins( $tcpdf_params['margins']['left'], $tcpdf_params['margins']['top'], $tcpdf_params['margins']['right'] );

	if ( $header_enable == 1 ) {
		$pdf->SetHeaderMargin( PDF_MARGIN_HEADER );
	}

	if ( $footer_enable == 1 ) {
		$pdf->SetFooterMargin( PDF_MARGIN_FOOTER );
	}

	// Set auto page breaks
	$pdf->SetAutoPageBreak( true, $tcpdf_params['margins']['bottom'] );

	// Set image scale factor
	if ( ! empty( $cert_args['ratio'] ) ) {
		$pdf->setImageScale( $cert_args['ratio'] );
	}

	// Set some language-dependent strings
	if ( isset( $l ) ) {
		$pdf->setLanguageArray( $l );
	}

	// Set fontsubsetting mode
	$pdf->setFontSubsetting( $subsetting );

	// Set font
	if ( ( ! empty( $font ) ) && ( ! empty( $font_size ) ) ) {
		$pdf->SetFont( $font, '', $font_size, true );
	}

	// Add a page
	$pdf->AddPage();

	// Added to let external manipulate the $pdf instance.
	/**
	 * Fires after setting certificate pdf data.
	 *
	 * @param \TCPDF $pdf `TCPDF` class instance.
	 * @param int $post_id Post ID.
	 *
	 * @since 2.4.7
	 *
	 */
	do_action( 'learndash_certification_after', $pdf, $cert_args['cert_id'] );

	// get featured image
	$img_file = automatorwp_learndash_get_thumbnail_path( $cert_args['cert_id'] );

	//Only print image if it exists
	if ( ! empty( $img_file ) ) {

		//Print BG image
		$pdf->setPrintHeader( false );

		// get the current page break margin
		$bMargin = $pdf->getBreakMargin();

		// get current auto-page-break mode
		$auto_page_break = $pdf->getAutoPageBreak();

		// disable auto-page-break
		$pdf->SetAutoPageBreak( false, 0 );

		// Get width and height of page for dynamic adjustments
		$pageH = $pdf->getPageHeight();
		$pageW = $pdf->getPageWidth();

		//Print the Background
		$pdf->Image( $img_file, '0', '0', $pageW, $pageH, '', '', '', false, 300, '', false, false, 0, false, false, false, false, array() );

		// restore auto-page-break status
		$pdf->SetAutoPageBreak( $auto_page_break, $bMargin );

		// set the starting point for the page content
		$pdf->setPageMark();
	}

	// Print post
	$pdf->writeHTMLCell( 0, 0, '', '', $cert_content, 0, 1, 0, true, '', true );

	// Set background
	$pdf->SetFillColor( 255, 255, 127 );
	$pdf->setCellPaddings( 0, 0, 0, 0 );
	// Print signature

	ob_clean();

	$full_path = $save_path . $file_name . '.pdf';

	switch ( $certificate_type ) {
		case 'quiz':
			$generated_cert = apply_filters( 'automatorwp_learndash_quiz_certificate', 'F' );
			break;
		case 'course':
			$generated_cert = apply_filters( 'automatorwp_learndash_course_certificate', 'F' );
			break;
		case 'automatorwp':
			$generated_cert = apply_filters( 'automatorwp_learndash_certificate', 'F' );
			break;
		default:
			$generated_cert = apply_filters( 'automatorwp_learndash_quiz_certificate', 'I' );
			break;

	}

	$pdf->Output( $full_path, $generated_cert ); 

	return array(
		'return'  => true,
		'message' => $full_path,
	);

}

/**
 * Helper function to complete certificate content
 *
 * @since 1.0.0
 * 
 * @param string 	$cert_content	Certificate content
 * @param array 	$args			Cetificate args
 *
 * @return mixed
 */
function automatorwp_learndash_certificate_content( $cert_content, $args, $course_id ) {
	
	$user = $args['user'];
	$completion_time = current_time( 'timestamp' );
	$format = 'F d, Y';

	preg_match( '/\[courseinfo(.*?)(completed_on)(.*?)\]/', $cert_content, $match );

	if ( $match && is_array( $match ) ) {
		$text = $match[0];
		$date_format = automatorwp_learndash_shorcode_attributes( 'courseinfo', $text );
		if ( $date_format ) {
			$format = key_exists( 'format', $date_format ) ? $date_format['format'] : $format;
		}
	}

	$cert_content = preg_replace( '/(\[courseinfo)/', '[courseinfo course_id="' . $course_id . '" ', $cert_content );
	$cert_content = preg_replace( '/\[courseinfo(.*?)(completed_on)(.*?)\]/', date_i18n( $format, $completion_time ), $cert_content );
	$cert_content = preg_replace( '/(\[usermeta)/', '[usermeta user_id="' . $user->ID . '" ', $cert_content );

	return apply_filters( 'automatorwp_learndash_certificate_content', $cert_content, $args, $course_id );

}

/**
 * Helper function to get attributes from shortcode
 *
 * @since 1.0.0
 *
 * @param string $tag 	Shortcode tag
 * @param string $text 	Shortcodes text
 *
 * @return array  $attributes
 */
function automatorwp_learndash_shorcode_attributes( $tag, $text ) {

	$attributes = array();

	preg_match_all( '/' . get_shortcode_regex() . '/s', $text, $match );
	
	if ( isset( $match[2] ) ) {
		foreach ( (array) $match[2] as $key => $value ) {
			if ( $tag === $value ) {
				$attributes = shortcode_parse_atts( $match[3][$key] );
			}
		}
	}

	return $attributes;
}

/**
 * Helper function to get thumbnail path
 *
 * @since 1.0.0
 *
 * @param int post_id	post ID
 *
 * @return string
 */
function automatorwp_learndash_get_thumbnail_path( $post_id ) {

	$thumbnail_id = get_post_meta( $post_id, '_thumbnail_id', true );

	if ( $thumbnail_id ) {
		$img_path      = get_post_meta( $thumbnail_id, '_wp_attached_file', true );
		$upload_url    = wp_upload_dir();
		$full_path = $upload_url['basedir'] . '/' . $img_path;

		return $full_path;
	}

}