<?php
/**
 * User Has Completed Course
 *
 * @package     AutomatorWP\Integrations\LearnDash\Filters\User_Has_Completed_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_User_Has_Completed_Course_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'learndash';
    public $filter = 'learndash_user_has_completed_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User has completed a course', 'automatorwp-pro' ),
            'select_option'     => __( 'User has completed <strong>a course</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}'  ),
            /* translators: %1$s: Group. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Course:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any course', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type' => 'sfwd-courses',
                    'default' => 'any'
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $course = $filter_options['post'];

        if ($course !== 'any') {
            $course_name = get_post($course)->post_title;
        } else{
            $course_name = 'any course';
        }
        
        // Bail if wrong configured
        if( empty( $course ) ) {
            $this->result = __( 'Filter not passed. course option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        if ( $course === 'any' ) {
            
            // Get all Learndash courses
            $all_courses = get_posts([
                'post_type' => 'sfwd-courses',
                'post_status' => 'publish',
                'numberposts' => -1
              ]);
            
            foreach ( $all_courses as $course_post ) {
                $course_is_completed = learndash_course_completed( $user_id, $course_post->ID );

                if ( (bool) $course_is_completed !== false ) {
                    break;
                }
            }

        } else {

            $course_is_completed = learndash_course_completed( $user_id, $course );

        }

        // Don't deserve if user has not completed course
        if( (bool) $course_is_completed === false) {
            $this->result = sprintf( __( 'Filter not passed. User has not completed course "%1$s".', 'automatorwp-pro' ),
                $course_name
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User has completed course "%1$s".', 'automatorwp-pro' ),
            $course_name
        );

        return $deserves_filter;

    }

}

new AutomatorWP_LearnDash_User_Has_Completed_Course_Filter();