<?php
/**
 * User Group
 *
 * @package     AutomatorWP\Integrations\LearnDash\Actions\User_Group
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_User_Group extends AutomatorWP_Integration_Action {

    public $integration = 'learndash';
    public $action = 'learndash_user_group';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add or remove user from a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Add or remove user from <strong>a group</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'add'     => __( 'Add', 'automatorwp-pro' ),
                                'remove'    => __( 'Remove', 'automatorwp-pro' ),
                            ),
                            'default' => 'add'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all groups', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'post_type'         => 'groups',
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'remove' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{post}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $group_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'add';
        }

        $groups = array();

        // Check specific group
        if( $group_id !== 'any' ) {

            $group = get_post( $group_id );

            // Bail if group doesn't exists
            if( ! $group ) {
                return;
            }

            $groups = array( $group_id );

        }

        switch ( $operation ) {
            case 'add':
                // If adding to all groups, get all groups
                if( $group_id === 'any' ) {
                    $groups = learndash_get_groups( true );
                }
                break;
            case 'remove':
                // If removing from all groups, get all user groups
                if( $group_id === 'any' ) {
                    $groups = learndash_get_users_group_ids( $user_id );
                }
                break;
        }

        // Add or remove user from groups
        foreach( $groups as $group_id ) {
            ld_update_group_access( $user_id, $group_id, ( $operation === 'remove' ) );
        }

    }

}

new AutomatorWP_LearnDash_User_Group();