<?php
/**
 * Complete Content Type
 *
 * @package     AutomatorWP\Integrations\H5P\Triggers\Complete_Content_Type
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_H5P_Complete_Content_Type extends AutomatorWP_Integration_Trigger {

    public $integration = 'h5p';
    public $trigger = 'h5p_complete_content_type';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a content of a specific type', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a content of <strong>a specific type</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Content type title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes a content of %1$s type %2$s time(s)', 'automatorwp-pro' ), '{content_type}', '{times}' ),
            /* translators: %1$s: Content type title. */
            'log_label'         => sprintf( __( 'User completes a content of %1$s type', 'automatorwp-pro' ), '{content_type}' ),
            'action'            => 'h5p_alter_user_result',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'content_type' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'content_type',
                    'name'              => __( 'Content Type:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_h5p_get_content_types',
                    'options_cb'        => 'automatorwp_h5p_options_cb_content_type',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param object    $data      Has the following properties score,max_score,opened,finished,time
     * @param int       $result_id  Only set if updating result
     * @param int       $content_id Identifier of the H5P Content
     * @param int       $user_id    Identifier of the User
     */
    public function listener( $data, $result_id, $content_id, $user_id ) {

        global $wpdb;

        // Get the library content type
        $content_type = $wpdb->get_var( $wpdb->prepare(
            "SELECT l.name
            FROM {$wpdb->prefix}h5p_contents c
            JOIN {$wpdb->prefix}h5p_libraries l ON l.id = c.library_id
            WHERE c.id = %d",
            $content_id
        ) );

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'content_id'    => $content_id,
            'content_type'  => $content_type,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if content type is not received
        if( ! isset( $event['content_type'] ) ) {
            return false;
        }

        $content_type = $event['content_type'];

        // Don't deserve if content type doesn't exists
        if( empty( $content_type ) ) {
            return false;
        }

        $required_content_type = $trigger_options['content_type'];

        // Don't deserve if content type doesn't match with the trigger option
        if( $required_content_type !== 'any' && $content_type !== $required_content_type ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_H5P_Complete_Content_Type();