<?php

/**
 * Place Added Category
 *
 * @package     AutomatorWP\Integrations\GeoDirectory\Triggers\Place_Added_Category
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class AutomatorWP_GeoDirectory_Place_Added_Category extends AutomatorWP_Integration_Trigger
{

    public $integration = 'geodirectory';
    public $trigger = 'geodirectory_place_added_category';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register()
    {

        automatorwp_register_trigger($this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __('User adds place to a category', 'automatorwp-pro'),
            'select_option'     => __('User adds <strong>place</strong> to a <strong>category</strong>', 'automatorwp-pro'),
            /* translators: %1$s: Place name. %2$s: The number of times. */
            'edit_label'        => sprintf(__('User adds place to %1$s %2$s time(s)', 'automatorwp-pro'), '{term}', '{times}'),
            /* translators: %1$s: Place name. %2$s: The number of times. */
            'log_label'         => sprintf(__('User adds place to %1$s', 'automatorwp-pro'), '{term}' ),
            'action'            => 'geodir_post_saved',
            'function'          => array($this, 'listener'),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'taxonomy' => 'gd_placecategory',
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags'  => array_merge(
                automatorwp_utilities_post_tags(),
            ) 

        ));
    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array $postarr, $gd_post, $post, $update
     */
    public function listener( $postarr, $gd_post, $post, $update )
    {

        $user_id = get_current_user_id();
        if ($user_id === 0) {
            return;
        }

        $post_id = $gd_post['ID'];
        $place_categories = $gd_post['tax_input']['gd_placecategory'];

        // To get the ID terms
        $terms_ids = automatorwp_get_term_ids( $post_id, 'gd_placecategory' );

        // Bail if post isn't assigned to any category
        if( empty( $terms_ids ) ) {
            return;
        }

        automatorwp_trigger_event(array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $post_id,
            'categories'    => $place_categories,
            'terms_ids'     => $terms_ids,
        ));
    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger($deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation)
    {

        // Don't deserve if post and terms IDs are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;
    }

}


new AutomatorWP_GeoDirectory_Place_Added_Category();
