<?php
/**
 * User Rank
 *
 * @package     AutomatorWP\Integrations\GamiPress\Filters\User_Rank
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_User_Rank_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'gamipress';
    public $filter = 'gamipress_user_rank';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User rank', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>rank</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Rank. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{post}'  ),
            /* translators: %1$s: Condition. %2$s: Rank. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{post}' ),
            'options'           => array(
                'condition' => automatorwp_utilities_number_condition_option(),
                'post' => automatorwp_gamipress_utilities_post_option( array(
                    'name'                  => __( 'Rank:', 'automatorwp-pro' ),
                    'option_default'        => __( 'Choose a rank', 'automatorwp-pro' ),
                    'option_none'           => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Rank ID', 'automatorwp-pro' ),
                    'default'               => '',
                    'post_type_cb'          => 'gamipress_get_rank_types_slugs'
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $condition = $filter_options['condition'];
        $post_id = absint( $filter_options['post'] );

        // Bail if post id not provided
        if( $post_id === 0 ) {
            $this->result = __( 'Filter not passed. Rank option is not configured.', 'automatorwp-pro' );
            return false;
        }

        $post = get_post( $post_id );

        // Bail if post doesn't exists
        if( ! $post ) {
            $this->result = __( 'Filter not passed. Rank configured does not exists.', 'automatorwp-pro' );
            return false;
        }

        // Bail if not is a rank
        if( ! in_array( $post->post_type, gamipress_get_rank_types_slugs() ) ) {
            $this->result = __( 'Filter not passed. Rank configured is not a GamiPress rank.', 'automatorwp-pro' );
            return false;
        }

        $user_rank = gamipress_get_user_rank( $user_id, $post->post_type );
        $rank_priority = gamipress_get_rank_priority( $post_id );
        $user_rank_priority = gamipress_get_rank_priority( $user_rank->ID );

        // Don't deserve if user rank does not matches with this condition
        if( ! automatorwp_number_condition_matches( $user_rank_priority, $rank_priority, $condition ) ) {
            $this->result = sprintf( __( 'Filter not passed. User rank %1$s does not meets the condition %2$s %3$s.', 'automatorwp-pro' ),
                $user_rank->post_title,
                automatorwp_utilities_get_condition_label( $condition ),
                $post->post_title
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User rank %1$s meets the condition %2$s %3$s.', 'automatorwp-pro' ),
            $user_rank->post_title,
            automatorwp_utilities_get_condition_label( $condition ),
            $post->post_title
        );

        return $deserves_filter;

    }

}

new AutomatorWP_GamiPress_User_Rank_Filter();