<?php
/**
 * Create Course
 *
 * @package     AutomatorWP\Integrations\FluentCommunity\Triggers\Create_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCommunity_Create_Course extends AutomatorWP_Integration_Trigger {

    public $integration = 'fluentcommunity';
    public $trigger = 'fluentcommunity_create_course';
    
    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {
    
        automatorwp_register_trigger($this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User creates a course', 'automatorwp-pro' ),
            'select_option'     => __( 'User creates a <strong>course</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Number of times. */
            'edit_label'        => sprintf( __( 'User creates a course %1$s time(s)', 'automatorwp-pro' ), '{times}' ),
            'log_label'         => __( 'User creates a course', 'automatorwp-pro' ),
            'action'            => 'fluent_community/course/created',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ));
    }
    
    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param object    $course 
     */
    public function listener( $course ) {

        $user_id = get_current_user_id();

        // Login is required
        if ( $user_id === 0 ) {
            return;
        }
 
        $course_data = $course->getOriginal();

        // Trigger event when user performs a comment action
        automatorwp_trigger_event(array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'course'    => $course_data['title'],
            'course_id' => $course_data['id'],
        ));
    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['course'] = ( isset( $event['course'] ) ? $event['course'] : '' );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['course'] = array(
            'name' => __( 'Course created', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_FluentCommunity_Create_Course();