<?php
/**
 * Publish Link Page
 *
 * @package     AutomatorWP\Integrations\Facebook\Actions\Publish_Link
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_Facebook_Publish_Link extends AutomatorWP_Integration_Action {
    
    public $integration = 'facebook';
    public $action = 'facebook_publish_link';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Share a link on a Facebook page', 'automatorwp-pro' ),
            'select_option' => __( 'Share a <strong>link</strong> to a Facebook page', 'automatorwp-pro' ),
            /* translators: %1$s: Facebook page. */
            'edit_label' => sprintf( __( 'Share a link on %1$s', 'automatorwp-pro' ), '{page}' ),
            /* translators: %1$s: Facebook page. */
            'log_label' => sprintf( __( 'Share a link on %1$s', 'automatorwp-pro' ), '{page}' ),
            'options' => array(
                'page' => array(
                    'from' => 'page',
                    'default' => __( 'a Facebook page', 'automatorwp-pro' ),
                    'fields' => array(
                        'page' => automatorwp_utilities_ajax_selector_field( array(
                            'field'             => 'page',
                            'option_default'    => __( 'page', 'automatorwp-pro' ),
                            'placeholder'       => __( 'Select a page', 'automatorwp-pro' ),
                            'name'              => __( 'Page:', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_facebook_get_pages',
                            'options_cb'        => 'automatorwp_facebook_options_cb_page',
                            'default'           => ''
                        ) ),
                        'link' => array(
                            'name' => __( 'Link:', 'automatorwp-pro' ),
                            'desc' => __( 'Enter the URL of the site to post on Facebook.', 'automatorwp-pro' ),
                            'type' => 'text',
                        ),
                        'message' => array(
                            'name' => __( 'Post:', 'automatorwp-pro' ),
                            'desc' => __( 'Enter the content for your post on Facebook. Please, be careful, this action might fail if executed in short intervals of time', 'automatorwp-pro' ),
                            'type' => 'textarea',
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){

        $message = '';

        // Bail if not selected page
        if( empty( $action_options['page'] ) ) {
            $this->result = __( 'No page selected on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Bail if not link
        if( empty( $action_options['link'] ) ) {
            $this->result = __( 'No link to post on Facebook page.', 'automatorwp-pro' );
            return;
        }

        if( !empty( $action_options['message'] ) ) {
            $message = $action_options['message'];
        }

        // Shorthand
        $page_id = $action_options['page'];
        $link = $action_options['link'];

        $params = automatorwp_facebook_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            return;
        }

        $api_version = automatorwp_facebook_api_version();
        $auth = get_option( 'automatorwp_facebook_auth' );
        $access_token = $auth['access_token'];
        
        // Get the page token
        $response = wp_remote_get('https://graph.facebook.com/' . $api_version . '/' . $page_id . '?fields=access_token&access_token=' . $access_token );
        $page_data = json_decode( wp_remote_retrieve_body( $response ), true );
        $page_token = $page_data['access_token'];

        $args = array(
            'body' => array(
                'message' => $message,
                'link'   => $link,
                'access_token'  => $page_token,
            )
        );

        // Publish the link on Facebook page
        $response = wp_remote_post( 'https://graph.facebook.com/' . $api_version . '/' . $page_id . '/feed', $args );

        $this->result = __( 'Link shared to Facebook successfully!', 'automatorwp-pro' );
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        $auth = get_option( 'automatorwp_facebook_auth' );

        // Warn user if the authorization has not been setup from settings
        if( empty ( $auth['access_token'] ) ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Facebook settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-facebook'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/facebook/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_Facebook_Publish_Link();