<?php
/**
 * Cancel Attendance
 *
 * @package     AutomatorWP\Integrations\Events_Manager\Triggers\Cancel_Attendance
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Events_Manager_Cancel_Attendance extends AutomatorWP_Integration_Trigger {

    public $integration = 'events_manager';
    public $trigger = 'events_manager_cancel_attendance';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User cancels attendance to an event', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>cancels attendance</strong> to an event', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User cancels attendance to %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'User cancels attendance to %1$s', 'automatorwp-pro' ), '{post}' ),
            'filter'            => 'em_booking_set_status',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Event:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any event', 'automatorwp-pro' ),
                    'post_type' => EM_POST_TYPE_EVENT
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param bool          $result
     * @param EM_Booking    $booking
     *
     * @return bool
     */
    public function listener( $result, $booking ) {

        // If status got changed from previously 'approved', then trigger cancel attendance event
        if ( $booking->booking_status != 1 && $booking->previous_status == 1 ) {

            $post_id = $booking->event->post_id;
            $user_id = $booking->person_id;

            // Trigger the cancel attendance
            automatorwp_trigger_event( array(
                'trigger' => $this->trigger,
                'user_id' => $user_id,
                'post_id' => $post_id,
            ) );
        }

        return $result;

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( $trigger_options['post'] !== 'any' && absint( $trigger_options['post'] ) !== absint( $event['post_id'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Events_Manager_Cancel_Attendance();