<?php
/**
 * Import CSV
 *
 * @package     AutomatorWP\Integrations\AutomatorWP\Triggers\Import_CSV
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_CSV_Import_CSV extends AutomatorWP_Integration_Trigger {

    public $integration = 'csv';
    public $trigger = 'csv_import_file';

    /**
     * The field conditions
     *
     * @since 1.0.0
     *
     * @var array $field_conditions
     */
    public $field_conditions = array();

    /**
     * The meta conditions
     *
     * @since 1.0.0
     *
     * @var array $meta_conditions
     */
    public $meta_conditions = array();

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        $fields = array();

        $fields = array_merge( $fields, array(
            'data_source' => array(
                'name' => __( 'Data source', 'automatorwp-pro' ),
                'type' => 'select',
                'options_cb' => 'automatorwp_csv_data_sources',
                'default' => 'upload'
            ),
            'csv' => array(
                'name' => __( 'CSV file:', 'automatorwp-pro' ),
                'desc' => __( 'The CSV file path.', 'automatorwp-pro' ),
                'type' => 'file',
                'query_args' => array(
                    'type' => 'text/csv',
                ),
                'classes' => 'automatorwp-hide-preview',
                'default' => ''
            ),
            'csv_text' => array(
                'name' => __( 'Data:', 'automatorwp-pro' ),
                'desc' => __( 'The CSV data.', 'automatorwp-pro' ),
                'type' => 'textarea',
                'default' => ''
            ),
        ) );

        $fields = apply_filters( 'automatorwp_import_csv_fields_after_csv_text', $fields );

        $fields = array_merge( $fields, array(
            'exclude_header' => array(
                'name' => __( 'Exclude header:', 'automatorwp-pro' ),
                'desc' => __( 'Enable to only get the rows with information, without headers.', 'automatorwp-pro' ),
                'type' => 'checkbox',
                'classes' => 'cmb2-switch',
                'default' => false
            ),
            'separator' => array(
                'name' => __( 'Separator:', 'automatorwp-pro' ),
                'desc' => __( 'Select the separator used to delimite fields in the CSV file. By default is comma', 'automatorwp-pro' ),
                'type' => 'select',
                'options' => array(
                    'comma'     => __( 'Comma', 'automatorwp-pro' ),
                    'semicolon' => __( 'Semicolon', 'automatorwp-pro' ),
                    'pipe'      => __( 'Pipe', 'automatorwp-pro' ),
                    'tab'       => __( 'Tab', 'automatorwp-pro' ),
                ),
                'default' => 'comma',
            ),
            'skip_rows' => array(
                'name' => __( 'Exclude starting rows:', 'automatorwp-pro' ),
                'desc' => __( 'Number of starting rows to ignore before reading data. Leave empty for no limit.', 'automatorwp-pro' ),
                'type' => 'text',
                'default' => '0',
            ),
            'limit_rows' => array(
                'name' => __( 'Limit rows:', 'automatorwp-pro' ),
                'desc' => __( 'Number of rows to import. Leave empty for no limit.', 'automatorwp-pro' ),
                'type' => 'text',
                'default' => '0',
            ),
        ) );

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'Import a CSV', 'automatorwp-pro' ),
            'select_option'     => '',
            'show_in_selector'  => false,
            /* translators: %1$s: Post title. */
            'edit_label'        => sprintf( __( 'Import a %1$s', 'automatorwp-pro' ), '{csv}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'Import a %1$s', 'automatorwp-pro' ), '{csv}' ),
            'options'           => array(
                'csv' => array(
                    'default' => __( 'csv', 'automatorwp-pro' ),
                    'fields' => $fields
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_import_file_get_csv_tags(),
            )
        ) );

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        parent::hooks();

        add_filter( 'automatorwp_get_import-file_automation_sql', array( $this, 'get_lines' ), 10, 7 );

        // Register dynamic trigger tags
        add_filter( 'automatorwp_trigger_tags', array( $this, 'dynamic_tags' ), 10, 2 );

        // Override edit item HTML
        add_filter( 'automatorwp_automation_item_edit_html_integration', array( $this, 'override_item_edit_html_integration' ), 10, 4 );
        add_filter( 'automatorwp_automations_column_triggers_integration', array( $this, 'override_item_edit_html_integration' ), 10, 4 );
        add_filter( 'automatorwp_get_log_integration_icon_integration', array( $this, 'override_item_edit_html_integration' ), 10, 4 );
        add_filter( 'automatorwp_automation_item_edit_html_classes', array( $this, 'override_item_edit_html_classes' ), 10, 4 );
        add_filter( 'automatorwp_automation_item_edit_html_actions', array( $this, 'override_item_edit_html_actions' ), 10, 4 );

        // Times tag, register as "no user trigger"
        add_filter( 'automatorwp_get_trigger_tag_replacement_times_no_user_triggers', array( $this, 'add_type_to_array' ) );
        add_filter( 'automatorwp_get_trigger_last_completion_log_no_user_types', array( $this, 'add_type_to_array' ) );
        add_filter( 'automatorwp_get_action_last_completion_log_no_user_types', array( $this, 'add_type_to_array' ) );

    }

    /**
     * Available filter to override the import file automation query
     *
     * @since 1.0.0
     *
     * @param string    $sql                The SQL query
     * @param stdClass  $automation         The automation object
     * @param stdClass  $trigger            The trigger object
     * @param bool      $count              True if is looking for the SQL to count the number of users
     * @param array     $trigger_options    The trigger's stored options
     * @param int       $lines_per_loop     The automation lines per loop option
     * @param int       $loop               The current loop
     */
    public function get_lines( $sql, $automation, $trigger, $count, $trigger_options, $lines_per_loop, $loop ) {

        global $automatorwp_csv_data;
        global $automatorwp_run_automation_error;
        
        // Shorthand
        $url = $trigger_options['csv'];
        $csv_text = $trigger_options['csv_text'];
        $exclude_header = (bool) $trigger_options['exclude_header'];
        $data_source = $trigger_options['data_source'];
        $skip_rows = absint( $trigger_options['skip_rows'] );
        $limit_rows = absint( $trigger_options['limit_rows'] );
        $separator = $trigger_options['separator'];
      
        switch ( $data_source ){
            case 'upload':
                // Check if file field has been saved
                if ( empty( $url ) ){
                    $automatorwp_run_automation_error = __( 'No CSV data found, did you saved the trigger Import a CSV?', 'automatorwp-pro' );
                    automatorwp_run_automation_finished( $automation );
                    return false;
                }
                $data = automatorwp_csv_get_data_from_csv_file( $url, $exclude_header, $skip_rows, $limit_rows, $separator );
                break;
            case 'paste':
                // Check if text field has been saved
                if ( empty( $csv_text ) ){
                    $automatorwp_run_automation_error = __( 'No CSV data found, did you saved the trigger Import a CSV?', 'automatorwp-pro' );
                    automatorwp_run_automation_finished( $automation );
                    return false;
                }
                $data = automatorwp_csv_get_data_from_csv_data( $csv_text, $exclude_header, $skip_rows, $limit_rows, $separator );
                break;
            default:
                $data = apply_filters( 'automatorwp_import_files_get_data_custom_source', $exclude_header, $skip_rows, $limit_rows, $trigger_options );
                break;
        }

        $automatorwp_csv_data = $data;

        // Used to return the count
        if( $count ) {
            return count( $data );
        }

        $limited_data = array();

        $offset = $loop * $lines_per_loop;
        $limit = $offset + $lines_per_loop;

        foreach( $data as $i => $row ) {

            if( $i >= $offset && $i < $limit ) {
                $limited_data[$i] = $row;
            }

        }

        return $limited_data;
        
    }

    /**
     * Register trigger dynamic tags
     *
     * @since 1.0.0
     *
     * @param array     $tags       The trigger tags
     * @param stdClass  $trigger    The trigger object
     * @return mixed
     */
    public function dynamic_tags( $tags, $trigger ) {

        // Bail if trigger type don't match this trigger
        if( $trigger->type !== $this->trigger ) {
            return $tags;
        }

        $trigger_options = automatorwp_get_trigger_stored_options( $trigger->id );

        // Override some trigger options to get the file clean
        $trigger_options['exclude_header'] = false;
        $trigger_options['skip_rows'] = 0;
        $trigger_options['limit_rows'] = 0;

        $lines = $this->get_lines( '', null, $trigger, false, $trigger_options, 1, 0 );
        
        // Bail if no data to get columns
        if ( empty( $lines ) ) {
            return $tags;
        }

        $added = false;

        foreach( $lines as $row ) {

            foreach( $row as $column => $value ) {

                $tags['t:' . $trigger->id . ':' .'csv_column_value:' . chr($column+65)] = array(
                    'label' => sprintf( __( 'Column %s', 'automatorwp-pro' ), chr($column+65) ),
                    'type' => 'text',
                    'preview' => $value
                );

                $added = true;

            }

        }

        // If any tag added, reorder the tags to keep dynamic first
        if( $added ) {
            $csv_tags = automatorwp_import_file_get_csv_tags();

            foreach( $csv_tags as $tag => $args ) {
                unset( $tags['t:' . $trigger->id . ':' . $tag] );

                $tags['t:' . $trigger->id . ':' . $tag] = $args;
            }
        }
        
        return $tags;
        

    }

    public function override_item_edit_html_integration( $integration, $object, $item_type, $automation ) {

        if( $item_type === 'trigger' && $object->type === $this->trigger ) {
            $integration['icon'] = AUTOMATORWP_CSV_URL . 'assets/automatorwp-import-file-csv.svg';
        }

        return $integration;

    }

    public function override_item_edit_html_classes( $classes, $object, $item_type, $automation ) {

        if( $automation->type === 'import-file' && $object->type === $this->trigger ) {
            $classes[] = 'automatorwp-no-grab';
        }

        return $classes;

    }

    public function override_item_edit_html_actions( $actions, $object, $item_type, $automation ) {

        if( $automation->type === 'import-file' && $object->type === $this->trigger ) {
            unset( $actions['move-up'] );
            unset( $actions['move-down'] );
            unset( $actions['delete'] );
        }

        return $actions;

    }

    public function add_type_to_array( $no_user_triggers ) {

        $no_user_triggers[] = $this->trigger;

        return $no_user_triggers;

    }

}

new AutomatorWP_CSV_Import_CSV();