<?php
/**
 * Delete Subscriber List
 *
 * @package     AutomatorWP\Integrations\Campaign_Monitor\Actions\Delete_Subscriber_List
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Campaign_Monitor_Delete_Subscriber_List extends AutomatorWP_Integration_Action {

    public $integration = 'campaign_monitor';
    public $action = 'campaignmonitor_delete_subscriber_list';

    /**
     * Register the action
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Delete subscriber from list', 'automatorwp-pro' ),
            'select_option'     => __( 'Delete <strong>subscriber</strong> from <strong>list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Subscriber. %2$s: List name. */
            'edit_label'        => sprintf( __( 'Delete %1$s from %2$s', 'automatorwp-pro' ), '{subscriber}', '{list}' ),
            /* translators: %1$s: Subscriber. %2$s: List name. */
            'log_label'         => sprintf( __( 'Delete %1$s from %2$s', 'automatorwp-pro' ), '{subscriber}', '{list}' ),
            'options'           => array(
                'list' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'list',
                    'option_default'    => __( 'list', 'automatorwp-pro' ),
                    'name'              => __( 'List:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'action_cb'         => 'automatorwp_campaign_monitor_get_lists',
                    'options_cb'        => 'automatorwp_campaign_monitor_options_cb_list',
                    'placeholder'       => 'Select a list',
                    'default'           => ''
                ) ),
                'subscriber' => array(
                    'from' => 'email',
                    'default' => __( 'subscriber', 'automatorwp-pro' ),
                    'fields' => array(
                        'email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'desc' => __( 'The subscriber email address.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                    )
                )
            ),
        ));
    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options
     * @param stdClass  $automation         The automation object
     */
    public function execute($action, $user_id, $action_options, $automation) {

        // Shorthand
        $list_id = $action_options['list'];
        $subscriber_email = $action_options['email'];

        $this->result = '';

        // Bail if list is empty
        if ( empty ( $list_id ) ){
            $this->result = __( 'No list selected.', 'automatorwp-pro' );
            return;
        }

        // Bail if subscriber email is empty
        if ( empty( $subscriber_email ) ){
            $this->result = __( 'Subscriber email field is empty.', 'automatorwp-pro' );
            return;
        }

        $api = automatorwp_campaign_monitor_get_api();

        if (!$api) {
            $this->result = __( 'Campaign Monitor API credentials are not set.', 'automatorwp-pro' );
            return;
        }

        $response_get = automatorwp_campaign_monitor_get_subscriber( $list_id, $subscriber_email );
        $status_code_get = wp_remote_retrieve_response_code( $response_get );
        $response_body_get = json_decode( wp_remote_retrieve_body( $response_get ), true );

        if ( isset( $response_body_get['State'] ) && $response_body_get['State'] === 'Deleted' ) {
             $this->result = __( 'Subscriber already deleted from list.', 'automatorwp-pro' );
             return;
        }

        if ( $status_code_get !== 200 ) {
            $this->result = __( 'Subscriber not found or already deleted.', 'automatorwp-pro' );
        }

        $response = automatorwp_campaign_monitor_delete_subscriber( $list_id, $subscriber_email );
        $status_code = wp_remote_retrieve_response_code( $response );
        $response_body = json_decode( wp_remote_retrieve_body( $response ), true );
        
        if ($status_code === 200) {
            $this->result = sprintf( __( '%s deleted from list.', 'automatorwp-pro' ), $subscriber_email);
        } else {
            $this->result = __( 'Subscriber not found or already deleted.', 'automatorwp-pro' );
        }
    }

/**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Warn user if the authorization has not been setup from settings
        if( ! automatorwp_campaign_monitor_get_api() ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Campaign Monitor settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-campaign_monitor'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/campaign-monitor/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }    

}

new AutomatorWP_Campaign_Monitor_Delete_Subscriber_List();

