<?php
/**
 * Calculate Formula
 *
 * @package     AutomatorWP\Integrations\Calculator\Actions\Calculate_Formula
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Calculator_Get_Calculate_Formula extends AutomatorWP_Integration_Action {

    public $integration = 'calculator';
    public $action = 'calculator_calculate_formula';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Calculate formula', 'automatorwp-pro' ),
            'select_option'     => __( 'Calculate <strong>formula</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation. */
            'edit_label'        => sprintf( __( 'Calculate %1$s', 'automatorwp-pro' ), '{formula}' ),
            'log_label'         => __( 'Calculate formula', 'automatorwp-pro' ),
            'options'           => array(
                'formula' => array(
                    'from'      => 'formula',
                    'default'   => __( 'formula', 'automatorwp-pro' ),
                    'fields' => array(
                        'formula' => array(
                            'name'      => __( 'Formula:', 'automatorwp-pro' ),
                            'desc'      => __( 'Mathematical formula.', 'automatorwp-pro' )
                                . ' ' . '<a href="https://automatorwp.com/docs/calculator/calculate-formula/" target="_blank">' . __( 'More info', 'automatorwp-pro' ) . '</a>',
                            'type'      => 'text',
                            'required'  => true,
                            'default'   => ''
                        ),
                     ) )
            ),
            'tags'  => automatorwp_calculator_get_actions_formula_result_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {
        
        $this->result = '';
        
        // Shorthand
        $formula = $action_options['formula'];
        
        // Bail if not formula
        if ( empty( $formula ) ) {
            $this->result = __( 'Please, insert operation to be calculated', 'automatorwp-pro' );
            return;
        }

        $this->formula_result = automatorwp_calculator_calculate_result( $formula ); 
        
        $this->result = sprintf( __( '%s = %s', 'automatorwp-pro' ), $formula, $this->formula_result );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['formula_result'] = ( isset( $this->formula_result ) ? $this->formula_result : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Calculator_Get_Calculate_Formula();