<?php
/**
 * Add Group Notification
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Actions\Add_Group_Notification
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_Add_Group_Notification extends AutomatorWP_Integration_Action {

    public $integration = 'buddypress';
    public $action = 'buddypress_add_group_notification';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add a notification to all members of a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Add a <strong>notification</strong> to all members of a group', 'automatorwp-pro' ),
            /* translators: %1$s: Notification. %2$s: Group Title. */
            'edit_label'        => sprintf( __( 'Add a %1$s to all members of %2$s', 'automatorwp-pro' ), '{notification}', '{group}' ),
            /* translators: %1$s: Notification. %2$s: Group Title. */
            'log_label'         => sprintf( __( 'Add a %1$s to all members %2$s', 'automatorwp-pro' ), '{notification}', '{group}' ),
            'options'           => array(
                'notification' => array(
                    'default' => __( 'notification', 'automatorwp-pro' ),
                    'fields' => array(
                        'exclude_author' => array(
                            'name' => __( 'Exclude user who completes the automation:', 'automatorwp-pro' ),
                            'desc' => __( 'Do not add the notification to user who completes the automation.', 'automatorwp-pro' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch'
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Notification\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                        'link' => array(
                            'name' => __( 'Link (Optional):', 'automatorwp-pro' ),
                            'desc' => __( 'URL link associated to the notification. Used for RSS feeds.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),
                'group' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group',
                    'option_default'    => __( 'group', 'automatorwp-pro' ),
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddypress_get_groups',
                    'options_cb'        => 'automatorwp_buddypress_options_cb_group',
                    'placeholder'       => __( 'Select a group', 'automatorwp-pro' ),
                    'default'           => '',
                ) )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $exclude_author = (bool) $action_options['exclude_author'];

        global $wpdb;

        $this->result = '';

        // Bail if groups and notifications components are not active
        if ( ! bp_is_active( 'groups' ) && ! bp_is_active( 'notifications' ) ) {
            $this->result = __( 'Groups or notifications component is not active.', 'automatorwp-pro' );
            return;
        }

        // Setup the group
        $group_id = $action_options['group'];

        if( absint( $group_id ) === 0 ) {
            $this->result = __( 'Invalid group provided on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Get all members in the group
        $members = BP_Groups_Member::get_group_member_ids( $group_id );

        $table = buddypress()->notifications->table_name;

        // Insert a new BuddyPress data to store the notification content
        ct_setup_table( 'automatorwp_buddypress_data' );

        $buddypress_data_id = ct_insert_object( array(
            'object_id' => $action->id,
            'user_id' => $user_id,
            'name' => $this->action
        ) );

        if( $buddypress_data_id ) {
            ct_add_object_meta( $buddypress_data_id, 'content', $action_options['content'], true );
            ct_add_object_meta( $buddypress_data_id, 'link', $action_options['link'], true );
        }

        ct_reset_setup_table();

        // Chunk members in groups of 200
        foreach ( array_chunk( $members, 200 ) as $members_ids ) {

            $sql = array();

            foreach ( $members_ids as $member_id ) {

                // Do not add the notification to the user who completes the automation if the option has been checked
                if( $exclude_author && absint( $member_id ) === $user_id ) {
                    continue;
                }

                // Setup the notification data
                $notification = array(
                    'user_id'           => $member_id, // User who receives the notification
                    'item_id'           => $buddypress_data_id,
                    'secondary_item_id' => $user_id, // User who completed the automation
                    'component_name'    => 'automatorwp',
                    'component_action'  => 'automatorwp_buddypress_group_notification_v2',
                    'date_notified'     => bp_core_current_time(),
                    'is_new'            => 1,
                );

                // Generate the SQL for the notification
                $sql[] = $wpdb->prepare(
                    "( %d, %d, %d, %s, %s, %s, %d )",
                    $notification['user_id'],
                    $notification['item_id'],
                    $notification['secondary_item_id'],
                    $notification['component_name'],
                    $notification['component_action'],
                    $notification['date_notified'],
                    $notification['is_new']
                );

            }

            if ( empty( $sql ) ) {
                return;
            }

            $list = join( ',', $sql );

            // Bulk insert of notifications to reduce the number of queries
            $wpdb->query( "INSERT INTO {$table} (user_id, item_id, secondary_item_id, component_name, component_action, date_notified, is_new) values {$list}" );

        }

        $this->result = sprintf( __( '%d members notified successfully.', 'automatorwp-pro' ), count( $members ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Format notifications
        add_filter( 'bp_notifications_get_notifications_for_user', array( $this, 'format_notifications' ), 10, 8 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Filters the notification content for notifications created by plugins.
     * If your plugin extends the {@link BP_Component} class, you should use the
     * 'notification_callback' parameter in your extended
     * {@link BP_Component::setup_globals()} method instead.
     *
     * @since BuddyPress 1.9.0
     * @since BuddyPress 2.6.0 Added $component_action_name, $component_name, $id as parameters.
     *
     * @param string $content               Component action. Deprecated. Do not do checks against this! Use
     *                                      the 6th parameter instead - $component_action_name.
     * @param int    $item_id               Notification item ID.
     * @param int    $secondary_item_id     Notification secondary item ID.
     * @param int    $action_item_count     Number of notifications with the same action.
     * @param string $format                Format of return. Either 'string' or 'object'.
     * @param string $component_action_name Canonical notification action.
     * @param string $component_name        Notification component ID.
     * @param int    $id                    Notification ID.
     *
     * @return string|array If $format is 'string', return a string of the notification content.
     *                      If $format is 'object', return an array formatted like:
     *                      array( 'text' => 'CONTENT', 'link' => 'LINK' )
     */
    public function format_notifications( $content, $item_id, $secondary_item_id, $action_item_count, $format, $component_action_name, $component_name, $id ) {

        // New version
        if( $component_action_name === 'automatorwp_buddypress_group_notification_v2' ) {

            // Get the notification content directly from the metadata
            ct_setup_table( 'automatorwp_buddypress_data' );

            $notification_content = ct_get_object_meta( $item_id, 'content', true );
            $notification_link = ct_get_object_meta( $item_id, 'link', true );

            ct_reset_setup_table();

            // Check if content has been stored as metadata
            if( ! empty( $notification_content ) ) {

                if( $format === 'string' ) {
                    return $notification_content;
                } else if( $format === 'object' ) {
                    return array(
                        'text' => $notification_content,
                        'link' => $notification_link
                    );
                }

            }

            // Get the notification content directly from the metadata
            ct_setup_table( 'automatorwp_buddypress_data' );

            $buddypress_data = ct_get_object( $item_id );

            ct_reset_setup_table();

            // Reassign vars for backward compatibility
            if( $buddypress_data ) {
                $component_action_name = 'automatorwp_buddypress_group_notification';
                $item_id = $buddypress_data->object_id;
            }

        }

        // Backward compatibility
        if( $component_action_name === 'automatorwp_buddypress_group_notification' ) {

            $user_id = absint( $secondary_item_id );

            if( $user_id === 0 ) {
                $user_id = get_current_user_id();
            }

            $action = automatorwp_get_action_object( $item_id );

            // Bail if action can't be found
            if( ! $action ) {
                return $content;
            }

            $automation = automatorwp_get_action_automation( $action->id );

            // Get all action options to parse all replacements
            $action_options = automatorwp_get_action_stored_options( $action->id );

            foreach( $action_options as $option => $value ) {
                // Replace all tags by their replacements
                $action_options[$option] = automatorwp_parse_automation_tags( $automation->id, $user_id, $value );
            }

            if( $format === 'string' ) {
                return $action_options['content'];
            } else if( $format === 'object' ) {
                return array(
                    'text' => $action_options['content'],
                    'link' => $action_options['link']
                );
            }

        }

        return $content;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyPress_Add_Group_Notification();