<?php
/**
 * Tags
 *
 * @package     AutomatorWP\BuddyBoss\Tags
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Custom tags
 *
 * @since 1.0.0
 *
 * @param array $tags The global tags
 *
 * @return array
 */
function automatorwp_buddyboss_get_tags( $tags ) {

    global $wpdb;

    // Profile
    $tags['buddyboss_profile_fields'] = array(
        'label' => __( 'User Profile Data', 'automatorwp-pro' ),
        'tags'  => array(),
        'icon'  => AUTOMATORWP_BUDDYBOSS_URL . 'assets/buddyboss.svg',
    );

    $profile_url = ( function_exists('bp_get_members_root_slug') ? trailingslashit( bp_get_root_domain() . '/' . bp_get_members_root_slug() ) : get_option( 'home' ) . '/members/' );

    $tags['buddyboss_profile_fields']['tags']['buddyboss_user_profile_link'] = array(
        'label'     => __( 'User Profile Link', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => '<href="' . $profile_url . 'automatorwp">AutomatorWP</a>',
    );

    $tags['buddyboss_profile_fields']['tags']['buddyboss_user_profile_url'] = array(
        'label'     => __( 'User Profile URL', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => $profile_url . 'automatorwp',
    );

    $prefix = automatorwp_buddyboss_get_table_prefix();
    $profile_fields = $wpdb->get_results( "SELECT * FROM {$prefix}bp_xprofile_fields ORDER BY field_order ASC" );

    foreach( $profile_fields as $profile_field ) {

        // Get the preview from the field type object
        $preview = '';

        if( function_exists( 'bp_xprofile_create_field_type' ) ) {
            $type_obj = bp_xprofile_create_field_type( $profile_field->type );

            if( $type_obj ) {
                $preview = $type_obj->name;
            }
        }

        $tags['buddyboss_profile_fields']['tags']['buddyboss_profile_field:' . $profile_field->id] = array(
            'label'     => $profile_field->name,
            'type'      => 'text',
            'preview'   => $preview,
        );

        $tags['buddyboss_profile_fields']['tags']['buddyboss_profile_field_raw:' . $profile_field->id] = array(
            'label'     => $profile_field->name . ' ' . __( '(Raw)', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => $preview . ' ' . __( '(without apply any formatting)', 'automatorwp-pro' ),
        );

    }

    return $tags;

}
add_filter( 'automatorwp_get_tags', 'automatorwp_buddyboss_get_tags' );

/**
 * Filter tags displayed on the tag selector
 *
 * @since 1.0.0
 *
 * @param array     $tags       The tags
 * @param stdClass  $automation The automation object
 * @param stdClass  $object     The trigger/action object
 * @param string    $item_type  The item type (trigger|action)
 *
 * @return array
 */
function automatorwp_buddyboss_tags_selector_html_tags( $tags, $automation, $object, $item_type ) {

    // Remove tags on anonymous user action
    if( $automation->type === 'anonymous' && $object->type === 'automatorwp_anonymous_user' ) {
        if( isset( $tags['buddyboss_profile_fields'] ) ) {
            unset( $tags['buddyboss_profile_fields'] );
        }
    }

    return $tags;

}
add_filter( 'automatorwp_tags_selector_html_tags', 'automatorwp_buddyboss_tags_selector_html_tags', 10, 4 );

/**
 * Custom tags replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param int       $automation_id  The automation ID
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 *
 * @return string
 */
function automatorwp_buddyboss_get_tag_replacement( $replacement, $tag_name, $automation_id, $user_id, $content ) {

    switch( $tag_name ) {
        case 'buddyboss_user_profile_link':
            $replacement = '<a href="' . bp_core_get_user_domain( $user_id ) . '">' . bp_core_get_user_displayname( $user_id ) . '</a>';
            break;
        case 'buddyboss_user_profile_url':
            $replacement = bp_core_get_user_domain( $user_id );
            break;
    }

    if( automatorwp_starts_with( $tag_name, 'buddyboss_profile_field_raw' ) ) {
        // Raw field tag

        $field_id = absint( str_replace( 'buddyboss_profile_field_raw:', '', $tag_name ) );

        if( ! class_exists( 'BP_XProfile_ProfileData' ) ) {
            return $replacement;
        }

        $replacement = maybe_unserialize( BP_XProfile_ProfileData::get_value_byid( $field_id, $user_id ) );

        if ( is_array( $replacement ) ) {
            $replacement = implode( ', ', $replacement );
        }
    } else if( automatorwp_starts_with( $tag_name, 'buddyboss_profile_field' ) ) {
        // Field tag

        $field_id = absint( str_replace( 'buddyboss_profile_field:', '', $tag_name ) );

        if( ! function_exists( 'xprofile_get_field_data' ) ) {
            return $replacement;
        }

        $replacement = xprofile_get_field_data( $field_id, $user_id, 'comma' );
    }

    return $replacement;

}
add_filter( 'automatorwp_get_tag_replacement', 'automatorwp_buddyboss_get_tag_replacement', 10, 5 );

/**
 * Activity tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_activity_tags() {

    $tags = array(
        'activity_id' => array(
            'label'     => __( 'Activity ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '123',
        ),
        'activity_url' => array(
            'label'     => __( 'Activity URL', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'https://automatorwp.com/',
        ),
        'activity_content' => array(
            'label'     => __( 'Activity content', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => __( 'The activity content', 'automatorwp-pro' ),
        ),
        'activity_author_id' => array(
            'label'     => __( 'Activity Author ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '123',
        ),
    );

    $media_allowed = (int) get_option( 'bp_media_allowed_per_batch', 10 );

    for($i=1; $i <= $media_allowed; $i++ ) {
        $tags['activity_media_url_' . $i] = array(
            'label'     => sprintf( __( 'Activity Media URL #%d', 'automatorwp-pro' ), $i ),
            'type'      => 'string',
            'preview'   => 'https://automatorwp.com/media-' . $i,
        );
    }

    $video_allowed = (int) get_option( 'bp_video_allowed_per_batch', 10 );

    for($i=1; $i <= $video_allowed; $i++ ) {
        $tags['activity_video_url_' . $i] = array(
            'label'     => sprintf( __( 'Activity Video URL #%d', 'automatorwp-pro' ), $i ),
            'type'      => 'string',
            'preview'   => 'https://automatorwp.com/video-' . $i,
        );
    }

    $document_allowed = (int) get_option( 'bp_document_allowed_per_batch', 10 );

    for($i=1; $i <= $document_allowed; $i++ ) {
        $tags['activity_document_url_' . $i] = array(
            'label'     => sprintf( __( 'Activity Document URL #%d', 'automatorwp-pro' ), $i ),
            'type'      => 'string',
            'preview'   => 'https://automatorwp.com/document-' . $i,
        );
    }

    return $tags;

}

/**
 * Custom trigger activity tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_activity_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'activity_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'activity_id', true );
            break;
        case 'activity_url':
            $activity_id = absint( automatorwp_get_log_meta( $log->id, 'activity_id', true ) );

            $replacement = bp_activity_get_permalink( $activity_id );
            break;
        case 'activity_content':
            $replacement = automatorwp_get_log_meta( $log->id, 'activity_content', true );
            break;
        case 'activity_author_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'activity_author_id', true );
            break;
        case 'activity_comment_content':
            $replacement = automatorwp_get_log_meta( $log->id, 'activity_comment_content', true );
            break;
    }

    // Media URL tags
    if( automatorwp_starts_with( $tag_name, 'activity_media_url_' ) ) {
        $urls = automatorwp_get_log_meta( $log->id, 'activity_media_urls', true );

        // Subtract 1 since is an index position
        $i = absint( str_replace( 'activity_media_url_', '', $tag_name ) ) - 1;

        $replacement = ( is_array( $urls ) && isset( $urls[$i] ) ? $urls[$i] : '' );
    }

    // Video URL tags
    if( automatorwp_starts_with( $tag_name, 'activity_video_url_' ) ) {
        $urls = automatorwp_get_log_meta( $log->id, 'activity_video_urls', true );

        // Subtract 1 since is an index position
        $i = absint( str_replace( 'activity_video_url_', '', $tag_name ) ) - 1;

        $replacement = ( is_array( $urls ) && isset( $urls[$i] ) ? $urls[$i] : '' );
    }

    // Document URL tags
    if( automatorwp_starts_with( $tag_name, 'activity_document_url_' ) ) {
        $urls = automatorwp_get_log_meta( $log->id, 'activity_document_urls', true );

        // Subtract 1 since is an index position
        $i = absint( str_replace( 'activity_document_url_', '', $tag_name ) ) - 1;

        $replacement = ( is_array( $urls ) && isset( $urls[$i] ) ? $urls[$i] : '' );
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_activity_tag_replacement', 10, 6 );

/**
 * Group tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_group_tags() {

    $groups_url =  get_option( 'home' ) . '/groups';

    if( function_exists( 'bp_get_groups_directory_permalink' ) ) {
        $groups_url = bp_get_groups_directory_permalink();
    }

    return array(
        'group_id' => array(
            'label'     => __( 'Group ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '1',
        ),
        'group_name' => array(
            'label'     => __( 'Group name', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => __( 'My group', 'automatorwp-pro' ),
        ),
        'group_description' => array(
            'label'     => __( 'Group description', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => __( 'My group description', 'automatorwp-pro' ),
        ),
        'group_url' => array(
            'label'     => __( 'Group URL', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => $groups_url . '/my-group',
        ),
        'group_link' => array(
            'label'     => __( 'Group link', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => '<a href="' . $groups_url . '/my-group' . '">' . __( 'My group', 'automatorwp-pro' ) . '</a>',
        ),
    );

}

/**
 * Custom trigger group tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_group_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    $group_tags = automatorwp_buddyboss_get_group_tags();
    $group_tags = array_keys( $group_tags );

    if( ! in_array( $tag_name, $group_tags ) ) {
        return $replacement;
    }

    // Bail if groups module is not active
    if( ! function_exists( 'groups_get_group' ) ) {
        return $replacement;
    }

    $group_id = absint( automatorwp_get_log_meta( $log->id, 'group_id', true ) );

    // Bail if not group ID store
    if( $group_id === 0 ) {
        return $replacement;
    }

    $group = groups_get_group( $group_id );

    if( ! $group ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'group_id':
            $replacement = $group_id;
            break;
        case 'group_name':
            $replacement = $group->name;
            break;
        case 'group_description':
            $replacement = $group->description;
            break;
        case 'group_url':
            $replacement = bp_get_group_permalink( $group );
            break;
        case 'group_link':
            $replacement = '<a href="' . bp_get_group_permalink( $group ) . '">' . $group->name . '</a>';
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_group_tag_replacement', 10, 6 );

/**
 * Invitation tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_invitation_tags() {

    return array(
        'inviter_id' => array(
            'label'     => __( 'Inviter ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '1',
        ),
        'invited_id' => array(
            'label'     => __( 'Invited ID', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => '2',
        ),
    );

}

/**
 * Custom trigger invitation tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_invitation_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'inviter_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'inviter_id', true );
            break;
        case 'invited_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'invited_id', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_invitation_tag_replacement', 10, 6 );

/**
 * Friendship tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_friendship_tags() {

    return array(
        'friend_id' => array(
            'label'     => __( 'Friend ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '1',
        ),
    );

}

/**
 * Custom trigger friendship tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_friendship_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'friend_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'friend_id', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_friendship_tag_replacement', 10, 6 );

/**
 * Forums tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_forum_tags() {

    return array(
        'forum_id' => array(
            'label'     => __( 'Forum ID', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '1',
        ),
        'forum_title' => array(
            'label'     => __( 'Forum title', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'The forum title',
        ),
    );

}

/**
 * Custom trigger forum tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_forum_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'forum_id':
            $replacement = automatorwp_get_log_meta( $log->id, 'forum_id', true );
            break;
        case 'forum_title':
            $replacement = automatorwp_get_log_meta( $log->id, 'forum_title', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_forum_tag_replacement', 10, 6 );

/**
 * Topic tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_buddyboss_get_topic_tags() {

    return array(
        'topic_title' => array(
            'label'     => __( 'Topic title', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'The topic title',
        ),
        'topic_url' => array(
            'label'     => __( 'Topic URL', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'The topic URL',
        ),
    );

}

/**
 * Custom trigger topic tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $trigger        The trigger object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last trigger log object
 *
 * @return string
 */
function automatorwp_buddyboss_get_trigger_topic_tag_replacement( $replacement, $tag_name, $trigger, $user_id, $content, $log ) {


    $trigger_args = automatorwp_get_trigger( $trigger->type );

    // Skip if trigger is not from this integration
    if( $trigger_args['integration'] !== 'buddyboss' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'topic_title':
            $replacement = automatorwp_get_log_meta( $log->id, 'topic_title', true );
            break;
        case 'topic_url':
            $replacement = automatorwp_get_log_meta( $log->id, 'topic_url', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_trigger_tag_replacement', 'automatorwp_buddyboss_get_trigger_topic_tag_replacement', 10, 6 );