<?php
/**
 * User Member Type
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Filters\User_Member_Type
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_User_Member_Type_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'buddyboss';
    public $filter = 'buddyboss_user_member_type';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User profile type', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>profile type</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Role. */
            'edit_label'        => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{member_type}'  ),
            /* translators: %1$s: Condition. %2$s: Role. */
            'log_label'         => sprintf( __( '%1$s %2$s', 'automatorwp-pro' ), '{condition}', '{member_type}' ),
            'options'           => array(
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'             => __( 'is equal to', 'automatorwp-pro' ),
                                'not_equal'         => __( 'is not equal to', 'automatorwp-pro' ),
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
                'member_type' => array(
                    'from' => 'member_type',
                    'default' => 'any',
                    'fields' => array(
                        'member_type' => array(
                            'name' => __( 'Profile Type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_buddyboss_member_types_options_cb',
                            'option_none' => true,
                            'default' => 'any'
                        ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $condition = $filter_options['condition'];
        $member_type = $filter_options['member_type'];

        // Bail if wrong configured
        if( empty( $member_type ) ) {
            $this->result = __( 'Filter not passed. Profile type option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        if( ! function_exists( 'bp_get_member_type' ) ) {
            $this->result = __( 'Filter not passed. Could not access to the function "bp_get_member_type".', 'automatorwp-pro' );
            return false;
        }

        $user_member_type = bp_get_member_type( $user_id );

        switch( $condition ) {
            case 'equal':
                // Don't deserve if user is not in this member type
                if( $user_member_type !== $member_type ) {
                    $this->result = sprintf( __( 'Filter not passed. User profile type "%1$s" does not meets the condition %2$s "%3$s".', 'automatorwp-pro' ),
                        $user_member_type,
                        automatorwp_utilities_get_condition_label( $condition ),
                        $member_type
                    );
                    return false;
                }
                break;
            case 'not_equal':
                // Don't deserve if user is in this member type
                if( $user_member_type === $member_type ) {
                    $this->result = sprintf( __( 'Filter not passed. User profile type "%1$s" does not meets the condition %2$s "%3$s".', 'automatorwp-pro' ),
                        $user_member_type,
                        automatorwp_utilities_get_condition_label( $condition ),
                        $member_type
                    );
                    return false;
                }
                break;
        }

        $this->result = sprintf( __( 'Filter passed. User profile type "%1$s" meets the condition %2$s "%3$s".', 'automatorwp-pro' ),
            $user_member_type,
            automatorwp_utilities_get_condition_label( $condition ),
            $member_type
        );

        return $deserves_filter;

    }

}

new AutomatorWP_BuddyBoss_User_Member_Type_Filter();