<?php
/**
 * Create Graphic Card
 *
 * @package     AutomatorWP\Integrations\Blockli\Actions\Create_Graphic_Card
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Blockli_Create_Graphic_Card extends AutomatorWP_Integration_Action {

    public $integration = 'blockli';
    public $action = 'blockli_create_graphic_card';

    /**
     * The new inserted post ID
     *
     * @since 1.0.0
     *
     * @var int|WP_Error $post_id
     */
    public $post_id = 0;

    /**
     * The post meta
     *
     * @since 1.0.0
     *
     * @var array $post_meta
     */
    public $post_meta = array();
    
    
    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        $post_status_options = automatorwp_blockli_post_status();

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create a graphic card in a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Create <strong>a graphic card</strong> in a group', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Group. */
            'edit_label'        => sprintf( __( 'Create %1$s in %2$s', 'automatorwp-pro' ), '{post}', '{term}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'Create %1$s', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => array(
                    'default' => __( 'a graphic card', 'automatorwp-pro' ),
                    'fields' => array(
                        'post_title' => array(
                            'name' => __( 'Title:', 'automatorwp-pro' ),
                            'desc' => __( 'The graphic card title.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'post_status' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'desc' => __( 'The graphic card status. By default, "draft".', 'automatorwp-pro' )
                                . ' ' . automatorwp_toggleable_options_list( $post_status_options ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'deep_link_key' => array(
                            'name' => __( 'Deep link:', 'automatorwp-pro' ),
                            'desc' => __( 'The URL for specific content.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'metatext_1_key' => array(
                            'name' => __( 'Meta text 1:', 'automatorwp-pro' ),
                            'desc' => __( 'Additional info (price, step...) for the post card.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'metatext_2_key' => array(
                            'name' => __( 'Meta text 2:', 'automatorwp-pro' ),
                            'desc' => __( 'Additional info (price, step...) for the post card.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'metatext_3_key' => array(
                            'name' => __( 'Meta text 3:', 'automatorwp-pro' ),
                            'desc' => __( 'Additional info (price, step...) for the post card.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'background_image_key' => array(
                            'name' => __( 'Background image:', 'automatorwp-pro' ),
                            'desc' => __( 'The URL for background image.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '' 
                        ),
                    )
                ),
                'term'  => automatorwp_utilities_term_option( array(
                        'taxonomy' => 'graphic_cards_cat'
                    )
                ),
                
            ),
            
        ) ) ;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Shorthand
        $group = $action_options['term'];

        // Setup post data
        $post_data = wp_parse_args( $action_options, array(
            'post_title'    => '',
            'post_name'     => '',
            'post_type'     => 'graphic_cards',
            'post_status'   => 'draft',
            'post_date'     => '',
            'post_author'   => '',
            'menu_order'    => '0',
            'post_password' => '',
        ) );

        // Setup post meta data
        $post_meta_card = array(
            'deep_link_key'         => $action_options['deep_link_key'],
            'metatext_1_key'        => $action_options['metatext_1_key'],
            'metatext_2_key'        => $action_options['metatext_2_key'],
            'metatext_3_key'        => $action_options['metatext_3_key'],
            'background_image_key'  => $action_options['background_image_key'],
        );

        // Format post date
        if( ! empty( $post_data['post_date'] ) ) {
            $post_data['post_date'] = date( 'Y-m-d H:i:s', strtotime( $post_data['post_date'] ) );
        }

        // Format post date
        if( absint( $post_data['post_author'] ) === 0 ) {
            $post_data['post_author'] = $user_id;
        }

        // Insert the post
        $this->post_id = wp_insert_post( $post_data );

        // Add post to category
        wp_set_post_terms( $this->post_id, array($group), 'graphic_cards_cat');

        if( $this->post_id ) {

            if( is_array( $post_meta_card ) ) {

                foreach( $post_meta_card as $meta_key => $meta_value ) {

                    // Sanitize
                    $meta_key = sanitize_text_field( $meta_key );
                    $meta_value = sanitize_text_field( $meta_value );

                    // Update post meta
                    update_post_meta( $this->post_id, $meta_key, $meta_value );

                    $this->post_meta[$meta_key] = $meta_value;

                    // Update action options to be passed on upcoming hooks
                    $post_meta_card[$meta_key] = array(
                        'meta_key' => $meta_key,
                        'meta_value' => $meta_value,
                    );

                }

            }

            /**
             * Action triggered before the create new user action gets executed
             *
             * @since 1.2.6
             *
             * @param int       $post_id            The new post ID
             * @param stdClass  $action             The action object
             * @param int       $user_id            The user ID (user who triggered the automation)
             * @param array     $action_options     The action's stored options (with tags already passed, included on meta keys and values)
             * @param stdClass  $automation         The action's automation object
             */
            do_action( 'automatorwp_blockli_create_graphic_card_executed', $this->post_id, $action, $user_id, $action_options, $automation );

    }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log post ID
        add_filter( 'automatorwp_user_completed_action_post_id', array( $this, 'post_id' ), 10, 6 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log post ID
     *
     * @since 1.0.0
     *
     * @param int       $post_id            The post ID, by default 0
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return int
     */
    public function post_id( $post_id, $action, $user_id, $event, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $post_id;
        }

        if( $this->post_id ) {
            $post_id = $this->post_id;
        }

        return $post_id;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store post fields
        $post_fields = array(
            'post_title',
            'post_status',  
        );

        foreach( $post_fields as $post_field ) {
            $log_meta[$post_field] = $action_options[$post_field];
        }

        // Store post meta
        $log_meta['post_meta'] = $this->post_meta;

        // Store result
        if( $this->post_id ) {
            $log_meta['result'] = __( 'Graphic card created correctly', 'automatorwp-pro' );
        } else if( is_wp_error( $this->post_id ) ) {
            $log_meta['result'] = $this->post_id->get_error_message();
        } else {
            $log_meta['result'] = __( 'Could not create graphic card', 'automatorwp-pro' );
        }

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['post_info'] = array(
            'name' => __( 'Post Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about the graphic card created.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['post_title'] = array(
            'name'      => __( 'Title:', 'automatorwp-pro' ),
            'desc'      => __( 'The post title.', 'automatorwp-pro' ),
            'type'      => 'text',
        );
        $log_fields['post_meta'] = array(
            'name' => __( 'Post Meta:', 'automatorwp-pro' ),
            'desc' => __( 'The post meta values keyed by their post meta key.', 'automatorwp-pro' ),
            'type' => 'text',
        );
        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }
}

new AutomatorWP_Blockli_Create_Graphic_Card();