<?php
/**
 * Get Reply
 *
 * @package     AutomatorWP\Integrations\bbPress\Triggers\Get_Reply
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_bbPress_Get_Reply extends AutomatorWP_Integration_Trigger {

    public $integration = 'bbpress';
    public $trigger = 'bbpress_get_reply';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a reply on a topic', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>gets a reply</strong> on a topic', 'automatorwp-pro' ),
            /* translators: %1$s: Topic title. %2$s: Forum title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a reply on %1$s of %2$s %3$s time(s)', 'automatorwp-pro' ), '{topic}', '{forum}', '{times}' ),
            /* translators: %1$s: Topic title. %2$s: Forum title. */
            'log_label'         => sprintf( __( 'User gets a reply on %1$s of %2$s', 'automatorwp-pro' ), '{topic}', '{forum}' ),
            'action'            => 'bbp_new_reply',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 5,
            'options'           => array(
                'topic' => array(
                    'from' => 'topic',
                    'fields' => array(
                        'topic' => automatorwp_utilities_post_field( array(
                            'name' => __( 'Topic:', 'automatorwp-pro' ),
                            'option_none_label' => __( 'any topic', 'automatorwp-pro' ),
                            'post_type_cb' => 'bbp_get_topic_post_type'
                        ) )
                    )
                ),
                'forum' => array(
                    'from' => 'forum',
                    'fields' => array(
                        'forum' => automatorwp_utilities_post_field( array(
                            'name' => __( 'Forum:', 'automatorwp-pro' ),
                            'option_none_label' => __( 'any forum', 'automatorwp-pro' ),
                            'post_type_cb' => 'bbp_get_forum_post_type'
                        ) )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $reply_id
     * @param int $topic_id
     * @param int $forum_id
     * @param array $anonymous_data
     * @param int $reply_author
     */
    public function listener( $reply_id, $topic_id, $forum_id, $anonymous_data, $reply_author ) {

        $topic_author = absint( get_post_field( 'post_author', $topic_id ) );
        $user_id = $reply_author;

        // Bail if not author of this topic
        if( $topic_author === 0 ) {
            return;
        }

        // Bail if author added a reply to his own topic
        if( $topic_author === $user_id ) {
            return;
        }

        // Trigger the create a reply
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $topic_author,
            'post_id'       => $reply_id,
            'topic_id'      => $topic_id,
            'forum_id'      => $forum_id,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['topic_id'] ) && ! isset( $event['forum_id'] ) ) {
            return false;
        }

        // Don't deserve if topic doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['topic_id'], $trigger_options['topic'] ) ) {
            return false;
        }

        // Don't deserve if forum doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['forum_id'], $trigger_options['forum'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_bbPress_Get_Reply();