<?php
/**
 * Create New Topic
 *
 * @package     AutomatorWP\Integrations\bbPress\Actions\Create_New_Topic
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_bbPress_Create_New_Topic extends AutomatorWP_Integration_Action {

    public $integration = 'bbpress';
    public $action = 'bbpress_create_new_topic';

    /**
     * The new inserted post ID
     *
     * @since 1.0.0
     *
     * @var int|WP_Error $post_id
     */
    public $post_id = 0;

    /**
     * The post meta
     *
     * @since 1.0.0
     *
     * @var array $post_meta
     */
    public $post_meta = array();

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log post ID
        add_filter( 'automatorwp_user_completed_action_post_id', array( $this, 'post_id' ), 10, 6 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        $post_status_options = array();

        if( function_exists('bbp_get_topic_statuses') ) {
            $statuses = bbp_get_topic_statuses();
        } else {
            $statuses = array(
                'publish' => __( 'Open', 'automatorwp-pro' ),
                'closed' => __( 'Closed', 'automatorwp-pro' ),
                'spam' => __( 'Spam', 'automatorwp-pro' ),
                'trash' => __( 'Trash', 'automatorwp-pro' ),
                'pending' => __( 'Pending', 'automatorwp-pro' ),
            );
        }

        foreach( $statuses as $post_status => $post_status_label ) {
            /* translators: %1$s: Post status key (draft, pending). %2$s: Post status label (Draft, Pending Review). */
            $post_status_options[] = sprintf( __( '<code>%1$s</code> for %2$s', 'automatorwp-pro' ), $post_status, $post_status_label );
        }

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create a topic in a forum', 'automatorwp-pro' ),
            'select_option'     => __( 'Create <strong>a topic</strong> in a forum', 'automatorwp-pro' ),
            /* translators: %1$s: Post. %2$s: Post Parent. */
            'edit_label'        => sprintf( __( 'Create %1$s in %2$s', 'automatorwp-pro' ), '{post}', '{post_parent}' ),
            /* translators: %1$s: Post. %2$s: Post Parent. */
            'log_label'         => sprintf( __( 'Create %1$s in %2$s', 'automatorwp-pro' ), '{post}', '{post_parent}' ),
            'options'           => array(
                'post' => array(
                    'default' => __( 'a topic', 'automatorwp-pro' ),
                    'fields' => array(
                        'post_title' => array(
                            'name' => __( 'Title:', 'automatorwp-pro' ),
                            'desc' => __( 'The topic title.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'post_name' => array(
                            'name' => __( 'URL slug:', 'automatorwp-pro' ),
                            'desc' => __( 'The last part of the URL. Leave blank to generate one based on the title.', 'automatorwp-pro' )
                                . ' ' . sprintf( __( '<a href="" target="_blank">Read about permalinks</a>', 'automatorwp-pro' ), 'https://bbpress.org/support/article/writing-posts/#post-field-descriptions' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_status' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'desc' => __( 'The topic status. By default, "pending".', 'automatorwp-pro' )
                                . ' ' . automatorwp_toggleable_options_list( $post_status_options ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_date' => array(
                            'name' => __( 'Date:', 'automatorwp-pro' ),
                            'desc' => __( 'The date of the topic. Supports "YYYY-MM-DD HH:MM:SS" and "YYYY-MM-DD" formats. By default, the date at the moment the automation gets completed.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_author' => array(
                            'name' => __( 'Author:', 'automatorwp-pro' ),
                            'desc' => __( 'The ID of the user who created this topic. By default, ID of user who completes this automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'The topic content. By default, empty.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                        'menu_order' => array(
                            'name' => __( 'Menu order:', 'automatorwp-pro' ),
                            'desc' => __( 'The topic menu order. By default, 0.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_password' => array(
                            'name' => __( 'Password:', 'automatorwp-pro' ),
                            'desc' => __( 'The password to access this topic. By default, empty.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'post_meta' => array(
                            'name' => __( 'Topic Meta:', 'automatorwp-pro' ),
                            'desc' => __( 'The topic meta values keyed by their topic meta key.', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table',
                            'options'     => array(
                                'add_button'        => __( 'Add meta', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'meta_key' => array(
                                    'name' => __( 'Meta Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'meta_value' => array(
                                    'name' => __( 'Meta Value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                    )
                ),
                'post_parent' => array(
                    'default' => __( 'a forum', 'automatorwp-pro' ),
                    'from' => 'post_parent',
                    'fields' => array(
                        'post_parent' => automatorwp_utilities_post_field( array(
                            'name'                  => __( 'Forum:', 'automatorwp-pro' ),
                            'desc'                  => __( 'The topic forum.', 'automatorwp-pro' ),
                            'option_none'           => true,
                            'option_none_label'     => __( 'a forum', 'automatorwp-pro' ),
                            'option_custom'         => true,
                            'option_custom_desc'    => __( 'Forum ID', 'automatorwp-pro' ),
                            'post_type_cb'          => 'bbp_get_forum_post_type'
                        ) ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Setup post data
        $post_data = wp_parse_args( $action_options, array(
            'post_title'    => '',
            'post_name'     => '',
            'post_type'     => bbp_get_topic_post_type(),
            'post_status'   => 'pending',
            'post_date'     => '',
            'post_author'   => '',
            'post_content'  => '',
            'post_excerpt'  => '',
            'post_parent'   => '',
            'menu_order'    => '0',
            'post_password' => '',
        ) );

        // Format post date
        if( ! empty( $post_data['post_date'] ) ) {
            $post_data['post_date'] = date( 'Y-m-d H:i:s', strtotime( $post_data['post_date'] ) );
        }

        // Format post date
        if( absint( $post_data['post_author'] ) === 0 ) {
            $post_data['post_author'] = $user_id;
        }

        // Format post parent
        if( $post_data['post_parent'] === 'any' ) {
            $post_data['post_parent'] = 0;
        }

        if( absint( $post_data['post_parent'] ) === 0 ) {
            $this->result = __( 'Could not create topic, forum not assigned', 'automatorwp-pro' );
            return;
        }

        // Insert the post
        $this->post_id = bbp_insert_topic( $post_data );

        if( $this->post_id ) {

            if( is_array( $action_options['post_meta'] ) ) {

                foreach( $action_options['post_meta'] as $i => $meta ) {

                    // Parse automation tags replacements to both, key and value
                    $meta_key = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['meta_key'] );
                    $meta_value = automatorwp_parse_automation_tags( $automation->id, $user_id, $meta['meta_value'] );

                    // Sanitize
                    $meta_key = sanitize_text_field( $meta_key );
                    $meta_value = sanitize_text_field( $meta_value );

                    // Update post meta
                    update_post_meta( $this->post_id, $meta_key, $meta_value );

                    $this->post_meta[$meta_key] = $meta_value;

                    // Update action options to be passed on upcoming hooks
                    $action_options['post_meta'][$i] = array(
                        'meta_key' => $meta_key,
                        'meta_value' => $meta_value,
                    );

                }

            }

            /**
             * Action triggered before the create new user action gets executed
             *
             * @since 1.2.6
             *
             * @param int       $post_id            The new post ID
             * @param stdClass  $action             The action object
             * @param int       $user_id            The user ID (user who triggered the automation)
             * @param array     $action_options     The action's stored options (with tags already passed, included on meta keys and values)
             * @param stdClass  $automation         The action's automation object
             */
            do_action( 'automatorwp_bbpress_create_topic_executed', $this->post_id, $action, $user_id, $action_options, $automation );

        }

    }

    /**
     * Action custom log post ID
     *
     * @since 1.0.0
     *
     * @param int       $post_id            The post ID, by default 0
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return int
     */
    public function post_id( $post_id, $action, $user_id, $event, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $post_id;
        }

        if( $this->post_id ) {
            $post_id = $this->post_id;
        }

        return $post_id;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store post fields
        $post_fields = array(
            'post_title',
            'post_name',
            'post_type',
            'post_status',
            'post_date',
            'post_author',
            'post_content',
            'post_parent',
            'menu_order',
            'post_password',
        );

        foreach( $post_fields as $post_field ) {
            $log_meta[$post_field] = $action_options[$post_field];
        }

        // Store post meta
        $log_meta['post_meta'] = $this->post_meta;

        // Store result
        if( $this->post_id ) {
            $log_meta['result'] = __( 'Topic created successfully', 'automatorwp-pro' );
        } else if( is_wp_error( $this->post_id ) ) {
            $log_meta['result'] = $this->post_id->get_error_message();
        } else {
            if( empty( $this->result ) ) {
                $log_meta['result'] = __( 'Could not create topic', 'automatorwp-pro' );
            } else {
                $log_meta['result'] = $this->result;
            }
        }

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['post_info'] = array(
            'name' => __( 'Topic Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about the topic created.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['post_title'] = array(
            'name'      => __( 'Title:', 'automatorwp-pro' ),
            'desc'      => __( 'The topic title.', 'automatorwp-pro' ),
            'type'      => 'text',
        );

        $log_fields['post_name'] = array(
            'name'      => __( 'URL slug:', 'automatorwp-pro' ),
            'desc'      => __( 'The topic URL slug.', 'automatorwp-pro' ),
            'type'      => 'text',
        );

        $log_fields['post_status'] = array(
            'name' => __( 'Status:', 'automatorwp-pro' ),
            'desc' => __( 'The topic status.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_date'] = array(
            'name' => __( 'Date:', 'automatorwp-pro' ),
            'desc' => __( 'The date of the topic.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_author'] = array(
            'name' => __( 'Author:', 'automatorwp-pro' ),
            'desc' => __( 'The ID of the user who added this topic.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_content'] = array(
            'name' => __( 'Content:', 'automatorwp-pro' ),
            'desc' => __( 'The topic content.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_parent'] = array(
            'name' => __( 'Forum:', 'automatorwp-pro' ),
            'desc' => __( 'The topic forum.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['menu_order'] = array(
            'name' => __( 'Menu order:', 'automatorwp-pro' ),
            'desc' => __( 'The topic menu order.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_password'] = array(
            'name' => __( 'Password:', 'automatorwp-pro' ),
            'desc' => __( 'The password to access this topic.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['post_meta'] = array(
            'name' => __( 'Topic Meta:', 'automatorwp-pro' ),
            'desc' => __( 'The topic meta values keyed by their topic meta key.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_bbPress_Create_New_Topic();