<?php
/**
 * User Automation
 *
 * @package     AutomatorWP\Integrations\ActiveMember360\Actions\User_Automation
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_ActiveMember360_User_Automation extends AutomatorWP_Integration_Action {

    public $integration = 'activemember360';
    public $action = 'activemember360_user_automation';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Assign or remove automation to user', 'automatorwp-pro' ),
            'select_option'     => __( 'Assign or remove <strong>automation</strong> to user', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (assign or remove). %2$s: Automation name. */
            'edit_label'        => sprintf( __( '%1$s %2$s to user', 'automatorwp-pro' ), '{operation}', '{automation}' ),
            /* translators: %1$s: Operation (assign or remove). %2$s: Automation name. */
            'log_label'         => sprintf( __( '%1$s %2$s to user', 'automatorwp-pro' ), '{operation}', '{automation}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'assign'       => __( 'Assign', 'automatorwp-pro' ),
                                'remove'    => __( 'Remove', 'automatorwp-pro' ),
                            ),
                            'default' => 'assign'
                        ),
                    )
                ),
                'automation' => array(
                    'from' => 'automation',
                    'fields' => array(
                        'automation' => array(
                            'name' => __( 'Automation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'automations_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * Get automations options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function automations_options_cb() {

        $options = array(
            'any' => __( 'all automations', 'automatorwp-pro' ),
        );

        // Get site automations
        $automations = apply_filters( 'mbr/site_automations/get', NULL );

        if( ! empty( $automations ) ) {
            foreach( $automations as $automation_id => $automation_name ) {
                $options[$automation_id] = $automation_name;
            }
        }

        return $options;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $automation_id = $action_options['automation'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'assign';
        }

        $user = get_user_by('id', $user_id);
        $user_email = $user->user_email;

        // Check if user is a contact in ActiveCampaign
        $user_am360 = mbr_get_contact_by_email( $user_email );

        if ( empty( $user_am360 ) )  {
            return;
        }

        $automations = array();

        // Check specific automation
        if( $automation_id !== 'any' ) {

            $automations = array( $automation_id );

        } else if( $automation_id === 'any' ) {

            // Get all automations
            $all_automations = apply_filters( 'mbr/site_automations/get', NULL );

            if( ! empty( $all_automations ) ) {
                $automations = array_keys( $all_automations );
            }
        }

        // Setup the data to sync
        $data = array(
            'contact_data' => array(
                'email' => $user_am360['email']
            )
        );

        switch ( $operation ) {
            case 'assign':

                // Add automations to the user
                $data['assign_automations'] = $automations;

                break;
            case 'remove':

                // Remove automations to the user
                $data['remove_automations'] = $automations;

                break;
        }

        // Sync user account
        apply_filters( 'mbr/api/sync_ctla', $data );

    }

}

new AutomatorWP_ActiveMember360_User_Automation();