<?php
/**
 * Get Vote
 *
 * @package     AutomatorWP\Integrations\wpForo\Triggers\Get_Vote
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_wpForo_Get_Vote extends AutomatorWP_Integration_Trigger {

    public $integration = 'wpforo';
    public $trigger = 'wpforo_get_vote';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a vote up/down on a post of a topic of a forum', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>gets a vote up/down</strong> on a post of a topic of a forum', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (vote up or vote down) %2$s: Topic title. %3$s: Forum title. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a %1$s on a post of %2$s of %3$s %4$s time(s)', 'automatorwp-pro' ), '{operation}', '{topic}', '{forum}', '{times}' ),
            /* translators: %1$s: Operation (vote up or vote down) %2$s: Topic title. %3$s: Forum title. */
            'log_label'         => sprintf( __( 'User gets a %1$s on a post of %2$s of %3$s', 'automatorwp-pro' ), '{operation}', '{topic}', '{forum}' ),
            'action'            => 'wpforo_vote',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'up'       => __( 'vote up', 'automatorwp-pro' ),
                                'down'    => __( 'vote down', 'automatorwp-pro' ),
                            ),
                            'default' => 'up'
                        ),
                    )
                ),
                'topic' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'topic',
                    'name'              => __( 'Topic:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any topic', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_wpforo_get_topics',
                    'options_cb'        => 'automatorwp_wpforo_options_cb_topic',
                    'default'           => 'any'
                ) ),
                'forum' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'forum',
                    'name'              => __( 'Forum:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any forum', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_wpforo_get_forums',
                    'options_cb'        => 'automatorwp_wpforo_options_cb_forum',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_wpforo_topic_tags(),
                automatorwp_wpforo_forum_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int   $reaction (1 = vote up, -1 = vote down)
     * @param array $post
     * @param int   $user_id
     */
    public function listener( $reaction, $post, $user_id ) {

        $post_author = $post['userid'];

        // Bail if not user provided
        if( $post_author === 0 ) {
            return;
        }

        $reply_id = $post['postid'];
        $topic_id = strval( $post['topicid'] );
        $forum_id = strval( $post['forumid'] );

        // Get the current Board
        $board_id = WPF()->board->get_current( 'boardid' );

        // Boards other than the main one (ID=0) take data from other additional tables in the database
        if ( absint( $board_id ) !== 0 ){
            $forum_id = $board_id . '-' . $post['forumid'];
            $topic_id = $board_id . '-' . $post['topicid'];
        }

        // Trigger the create a reply
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $post_author,
            'post_id'       => $reply_id,
            'topic_id'      => $topic_id,
            'forum_id'      => $forum_id,
            'vote'          => ( $reaction === 1 ? 'up' : 'down' ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Shorthand
        $operation = $trigger_options['operation'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'up';
        }

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['topic_id'] ) && ! isset( $event['forum_id'] )  && ! isset( $event['vote'] ) ) {
            return false;
        }

        // Don't deserve if topic doesn't match with the trigger option
        if( $trigger_options['topic'] !== 'any' && $event['topic_id']  !== $trigger_options['topic'] ) {
            return false;
        }

        // Don't deserve if forum doesn't match with the trigger option
        if( $trigger_options['forum'] !== 'any' &&  $event['forum_id'] !==  $trigger_options['forum'] ) {
            return false;
        }

        // Don't deserve if user vote doesn't match with the trigger option
        if( $operation !== $event['vote'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['forum_id'] = ( isset( $event['forum_id'] ) ? $event['forum_id'] : 0 );
        $log_meta['topic_id'] = ( isset( $event['topic_id'] ) ? $event['topic_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_wpForo_Get_Vote();