<?php
/**
 * Get Rate Rating
 *
 * @package     AutomatorWP\Integrations\WP_PostRatings\Triggers\Get_Rate_Rating
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_PostRatings_Get_Rate_Rating extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_postratings';
    public $trigger = 'wp_postratings_get_rate_rating';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        // All public post types
        $post_types = get_post_types( array(
            'public' => true
        ) );

        // Remove keys
        $post_types = array_values( $post_types );

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a rate on a post with a rating greater than, less than or equal to a specific amount', 'automatorwp-pro' ),
            'select_option'     => __( 'User gets a rate on a post with a rating <strong>greater than, less than or equal</strong> to a specific amount', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Rating. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a rate on %1$s with a rating %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{post}', '{condition}', '{rating}', '{times}' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Rating. */
            'log_label'         => sprintf( __( 'User gets a rate on %1$s with a rating %2$s %3$s', 'automatorwp-pro' ), '{post}', '{condition}', '{rating}' ),
            'action'            => 'rate_post',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'post_type' => $post_types
                ) ),
                'condition' => automatorwp_utilities_condition_option(),
                'rating' => array(
                    'from' => 'rating',
                    'fields' => array(
                        'rating' => array(
                            'name' => __( 'Rating:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '1',
                            ),
                            'default' => 1
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $user_id
     * @param int $post_id
     * @param int $rating_value
     */
    public function listener( $user_id, $post_id, $rating_value ) {

        // Get the post author
        $user_id = absint( get_post_field( 'post_author', $post_id ) );

        // Bail if can't find the user ID
        if( $user_id === 0 ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'post_id'   => $post_id,
            'rating'    => $rating_value,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        $rating = absint( $event['rating'] );
        $required_rating = absint( $trigger_options['rating'] );

        // Don't deserve if rating doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $rating, $required_rating, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_PostRatings_Get_Rate_Rating();