<?php
/**
 * Job Not Filled
 *
 * @package     AutomatorWP\Integrations\WP_Job_Manager\Triggers\Job_Not_Filled
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Job_Manager_Job_Not_Filled extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_job_manager';
    public $trigger = 'wp_job_manager_job_not_filled';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User marks a job of a type as not filled', 'automatorwp-pro' ),
            'select_option'     => __( 'User marks a job of a type as <strong>not filled</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Term title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User marks a job of %1$s as not filled %2$s time(s)', 'automatorwp-pro' ), '{term}', '{times}' ),
            /* translators: %1$s: Term title. */
            'log_label'         => sprintf( __( 'User marks a job of %1$s as not filled', 'automatorwp-pro' ), '{term}' ),
            'action'            => 'job_manager_my_job_do_action',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'name'              => __( 'Type:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any type', 'automatorwp-pro' ),
                    'taxonomy'          => 'job_listing_type',
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags( __( 'Job', 'automatorwp-pro' ) ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param string    $action
     * @param int       $post_id
     */
    public function listener( $action, $post_id ) {

        // Bail if job is not being marked as not filled
        if( $action !== 'mark_not_filled' ) {
            return;
        }

        $post = get_post( $post_id );
        $user_id = $post->post_author;

        $terms_ids = automatorwp_get_term_ids( $post->ID, 'job_listing_type' );

        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'post_id'   => $post->ID,
            'terms_ids' => $terms_ids,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post and terms IDs are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_Job_Manager_Job_Not_Filled();