<?php
/**
 * Anonymous Message Received
 *
 * @package     AutomatorWP\Integrations\WhatsApp\Triggers\Anonymous_Message_Received
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WhatsApp_Anonymous_Message_Received extends AutomatorWP_Integration_Trigger {

    public $integration = 'whatsapp';
    public $trigger = 'whatsapp_anonymous_message_received';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            'label'             => __( 'Message received', 'automatorwp-pro' ),
            'select_option'     => __( '<strong>Message</strong> received', 'automatorwp-pro' ),
            'edit_label'        => __( 'Message received', 'automatorwp-pro' ),
            'log_label'         => __( 'Message received', 'automatorwp-pro' ),
            'action'            => 'automatorwp_whatsapp_message_received',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                // No options
            ),
            'tags' => array_merge(
                automatorwp_whatsapp_get_webhook_tags()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     * 
     * @param array     $params     Data received
     */
    public function listener( $params ) {

        $message = $params['entry'][0]['changes'][0]['value']['messages'][0];
        $contact_name = $params['entry'][0]['changes'][0]['value']['contacts'][0]['profile']['name'];
        $received_time = date('Y-m-d H:i:s', $message['timestamp']);

        $this->result = __( 'Received message. ', 'automatorwp-pro' );
  
        // Trigger the message is received
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'from'          => '+' . $message['from'],
            'received_time' => $received_time,
            'message'       => $message['text']['body'],
            'profile'       => $contact_name,
        ) );   
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
         add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_anonymous_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'trigger' || $object->type !== $this->trigger ) {
            return;
        }

        automatorwp_whatsapp_trigger_notice();

    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if trigger type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['from'] = ( isset( $event['from'] ) ? $event['from'] : '' );
        $log_meta['received_time'] = ( isset( $event['received_time'] ) ? $event['received_time'] : '' );
        $log_meta['message'] = ( isset( $event['message'] ) ? $event['message'] : '' );
        $log_meta['profile'] = ( isset( $event['profile'] ) ? $event['profile'] : '' );
        
        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this action
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['from'] = array(
            'name' => __( 'From:', 'automatorwp-pro' ),
            'type' => 'text',
        );
        $log_fields['received_time'] = array(
            'name' => __( 'Received time:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['message'] = array(
            'name' => __( 'Message:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['profile'] = array(
            'name' => __( 'Profile name:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

    
}

new AutomatorWP_WhatsApp_Anonymous_Message_Received();