<?php
/**
 * Ajax Functions
 *
 * @package     AutomatorWP\Integrations\WhatsApp\Ajax_Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;


/**
 * AJAX handler for the authorize action
 *
 * @since 1.0.0
 */
function automatorwp_whatsapp_ajax_authorize() {
    // Security check
    check_ajax_referer( 'automatorwp_admin', 'nonce' );

    $prefix = 'automatorwp_whatsapp_';

    $url = automatorwp_whatsapp_get_url();
    $token = sanitize_text_field( $_POST['token'] );
    $phone_id = sanitize_text_field( $_POST['phone_id'] );
    $business_id = sanitize_text_field( $_POST['business_id'] );
    $webhook_verification_id = sanitize_text_field( $_POST['webhook_verification_id'] );

    // Check parameters given
    if( empty( $token ) || empty( $phone_id ) || empty( $business_id ) || empty( $webhook_verification_id ) ) {
        wp_send_json_error( array( 'message' => __( 'API Token, Phone ID, Business ID and Webhook verification ID are required to connect with WhatsApp', 'automatorwp-pro' ) ) );
        return;
    }

    // To get first answer and check the connection with the phone id
    $response = wp_remote_get( $url . $phone_id . '/business_compliance_info', array(
        'headers' => array(
            'Authorization' => 'Bearer '. $token,
        )
    ) );

    // Incorrect API token or Phone ID
    if ( isset( $response['response']['code'] ) && $response['response']['code'] !== 200 ){
        wp_send_json_error ( array( 'message' => __( 'Please, check your credentials', 'automatorwp-pro' ) ) );
        return;
    }

    // To get second answer and check the connection with the business id
    $response = wp_remote_get( $url . $business_id, array(
        'headers' => array(
            'Authorization' => 'Bearer '. $token,
        )
    ) );

    if ( isset( $response['response']['code'] ) && $response['response']['code'] !== 200 ){
        wp_send_json_error (array( 'message' => __( 'Please, check your Business ID', 'automatorwp-pro' ) ) );
        return;
    }
    
    $settings = get_option( 'automatorwp_settings' );

    // Save API credentials
    $settings[$prefix . 'token'] = $token;
    $settings[$prefix . 'phone_id'] = $phone_id;
    $settings[$prefix . 'business_id'] = $business_id;
    $settings[$prefix . 'webhook_verification_id'] = $webhook_verification_id;

    // Update settings
    update_option( 'automatorwp_settings', $settings );

    $admin_url = admin_url( 'admin.php?page=automatorwp_settings&tab=opt-tab-whatsapp' );
   
    wp_send_json_success( array(
        'message' => __( 'Correct data to connect with WhatsApp', 'automatorwp-pro' ),
        'redirect_url' => $admin_url
    ) );

}
add_action( 'wp_ajax_automatorwp_whatsapp_authorize',  'automatorwp_whatsapp_ajax_authorize' );

/**
 * Set the default URL value
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_whatsapp_ajax_refresh( ) {
    // Security check
    check_ajax_referer( 'automatorwp_admin', 'nonce' );

    $prefix = 'automatorwp_whatsapp_';

    // Get random characters for slug
    $slug = strtolower( wp_generate_password( 8, false ) );

    $settings = get_option( 'automatorwp_settings' );

    $settings[$prefix . 'webhook'] = get_rest_url() . 'whatsapp/webhooks/' . $slug;
    $settings[$prefix . 'slug'] = $slug;
    
    update_option( 'automatorwp_settings', $settings);

    $admin_url = admin_url( 'admin.php?page=automatorwp_settings&tab=opt-tab-whatsapp' );
   
    wp_send_json_success( array(
        'message' => __( 'Webhook URL refreshed', 'automatorwp' ),
        'redirect_url' => $admin_url
    ) );
    
}

add_action( 'wp_ajax_automatorwp_whatsapp_refresh',  'automatorwp_whatsapp_ajax_refresh' );


/**
 * Ajax function for selecting templates
 *
 * @since 1.0.0
 */
function automatorwp_whatsapp_ajax_get_templates() {
    // Security check, forces to die if not security passed
    check_ajax_referer( 'automatorwp_admin', 'nonce' );

    global $wpdb;

    // Pull back the search string
    $search = isset( $_REQUEST['q'] ) ? $wpdb->esc_like( sanitize_text_field( $_REQUEST['q'] ) ) : '';

    $templates = automatorwp_whatsapp_get_templates();
    
    $results = array();

    // Parse templates results to match select2 results
    foreach ( $templates as $template ) {

        if( ! empty( $search ) ) {
            if( strpos( strtolower( $template['name'] ), strtolower( $search ) ) === false ) {
                continue;
            }
        }

        $results[] = array(
            'id'   => $template['name'],
            'text' => $template['name']
        );
    }

    // Prepend option none
    $results = automatorwp_ajax_parse_extra_options( $results );

    // Return our results
    wp_send_json_success( $results );
    die;

}
add_action( 'wp_ajax_automatorwp_whatsapp_get_templates', 'automatorwp_whatsapp_ajax_get_templates' );