<?php
/**
 * Logs
 *
 * @package     AutomatorWP\Webhooks\Custom_Tables\Logs
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Parse query args for logs
 *
 * @since 1.0.0
 *
 * @param string $where
 * @param CT_Query $ct_query
 *
 * @return string
 */
function automatorwp_webhooks_logs_query_where( $where, $ct_query ) {

    global $ct_table;

    if( $ct_table->name !== 'automatorwp_webhooks_logs' ) {
        return $where;
    }

    // Shorthand
    $qv = $ct_query->query_vars;

    // URL
    $where .= automatorwp_custom_table_where( $qv, 'url', 'string' );

    // Method
    $where .= automatorwp_custom_table_where( $qv, 'method', 'string' );

    // Logged-in user ID
    $where .= automatorwp_custom_table_where( $qv, 'logged_in_user_id', 'integer' );

    // User ID
    $where .= automatorwp_custom_table_where( $qv, 'user_id', 'integer' );

    return $where;
}
add_filter( 'ct_query_where', 'automatorwp_webhooks_logs_query_where', 10, 2 );

/**
 * Define the search fields for logs
 *
 * @since 1.0.0
 *
 * @param array $search_fields
 *
 * @return array
 */
function automatorwp_webhooks_logs_search_fields( $search_fields ) {

    $search_fields[] = 'url';

    return $search_fields;

}
add_filter( 'ct_query_automatorwp_webhooks_logs_search_fields', 'automatorwp_webhooks_logs_search_fields' );

/**
 * Columns for logs list view
 *
 * @since 1.0.0
 *
 * @param array $columns
 *
 * @return array
 */
function automatorwp_webhooks_manage_logs_columns( $columns = array() ) {

    $columns['url']                 = __( 'URL', 'automatorwp-pro' );
    $columns['method']              = __( 'Method', 'automatorwp-pro' );
    $columns['logged_in_user_id']   = __( 'Logged-in User', 'automatorwp-pro' );
    $columns['user_id']             = __( 'User', 'automatorwp-pro' );
    $columns['date']                = __( 'Date', 'automatorwp-pro' );

    return $columns;
}
add_filter( 'manage_automatorwp_webhooks_logs_columns', 'automatorwp_webhooks_manage_logs_columns' );

/**
 * Sortable columns for logs list view
 *
 * @since 1.0.0
 *
 * @param array $sortable_columns
 *
 * @return array
 */
function automatorwp_webhooks_manage_logs_sortable_columns( $sortable_columns ) {

    $sortable_columns['url']                = array( 'url', false );
    $sortable_columns['method']             = array( 'method', false );
    $sortable_columns['logged_in_user_id']  = array( 'logged_in_user_id', false );
    $sortable_columns['user_id']            = array( 'user_id', false );
    $sortable_columns['date']               = array( 'date', true );

    return $sortable_columns;

}
add_filter( 'manage_automatorwp_webhooks_logs_sortable_columns', 'automatorwp_webhooks_manage_logs_sortable_columns' );

/**
 * Columns rendering for logs list view
 *
 * @since  1.0.0
 *
 * @param string $column_name
 * @param integer $object_id
 */
function automatorwp_webhooks_manage_logs_custom_column(  $column_name, $object_id ) {

    // Setup vars
    $log = ct_get_object( $object_id );

    switch( $column_name ) {
        case 'url':
            ?>
            <strong><a href="<?php echo ct_get_edit_link( 'automatorwp_webhooks_logs', $log->id ); ?>"><?php echo $log->url; ?></a></strong>
            <?php

            break;
        case 'logged_in_user_id':
            $user = get_userdata( $log->logged_in_user_id );

            if( $user ) {

                if( current_user_can( 'edit_users' ) ) {
                    ?>

                    <a href="<?php echo get_edit_user_link( $user->ID ); ?>"><?php echo $user->display_name . ' (' . $user->user_login . ')'; ?></a>

                    <?php
                } else {
                    echo $user->display_name;
                }

            } else {
                echo __( 'No user assigned', 'automatorwp-pro' );
            }
            break;
        case 'user_id':
            $user = get_userdata( $log->user_id );

            if( $user ) {

                if( current_user_can( 'edit_users' ) ) {
                    ?>

                    <a href="<?php echo get_edit_user_link( $user->ID ); ?>"><?php echo $user->display_name . ' (' . $user->user_login . ')'; ?></a>

                    <?php
                } else {
                    echo $user->display_name;
                }

            } else {
                echo __( 'No user assigned', 'automatorwp-pro' );
            }
            break;
        case 'date':
            ?>

            <abbr title="<?php echo date( 'Y/m/d g:i:s a', strtotime( $log->date ) ); ?>"><?php echo date( 'Y/m/d', strtotime( $log->date ) ); ?></abbr>

            <?php
            break;
    }
}
add_action( 'manage_automatorwp_webhooks_logs_custom_column', 'automatorwp_webhooks_manage_logs_custom_column', 10, 2 );

/**
 * Logs meta boxes
 *
 * @since  1.0.0
 */
function automatorwp_webhooks_add_logs_meta_boxes() {

    remove_meta_box( 'submitdiv', 'automatorwp_webhooks_logs', 'side' );
    add_meta_box( 'automatorwp_log_details', __( 'Log Details', 'automatorwp-pro' ), 'automatorwp_webhooks_log_details_meta_box', 'automatorwp_webhooks_logs', 'side', 'default' );
    add_meta_box( 'automatorwp_log_data', __( 'Log Data', 'automatorwp-pro' ), 'automatorwp_webhooks_log_data_meta_box', 'automatorwp_webhooks_logs', 'normal', 'default' );

}
add_action( 'add_meta_boxes', 'automatorwp_webhooks_add_logs_meta_boxes' );

/**
 * Render the log details meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_webhooks_log_details_meta_box( $object = null ) {

    global $ct_table;

    // Setup vars
    $log_types = automatorwp_get_log_types();

    ?>
    <div class="submitbox" id="submitpost" style="margin: -6px -12px -12px;">

        <div id="minor-publishing">

            <div id="misc-publishing-actions">

                <div class="misc-pub-section misc-pub-post-user" id="user">
                    <?php $user = get_userdata( $object->logged_in_user_id ); ?>
                    <?php echo __( 'Logged-in User:', 'automatorwp-pro' ); ?> <span id="post-user-display"><?php
                        if( $user ) {
                            if( current_user_can( 'edit_users' ) ) {
                                ?>

                                <a href="<?php echo get_edit_user_link( $user->ID ); ?>"><?php echo $user->display_name . ' (' . $user->user_login . ')'; ?></a>

                                <?php
                            } else {
                                echo $user->display_name;
                            }
                        } else {
                            echo __( 'No user assigned', 'automatorwp-pro' );
                        } ?></span>
                </div>

                <div class="misc-pub-section misc-pub-post-user" id="user">
                    <?php $user = get_userdata( $object->user_id ); ?>
                    <?php echo __( 'User:', 'automatorwp-pro' ); ?> <span id="post-user-display"><?php
                        if( $user ) {
                            if( current_user_can( 'edit_users' ) ) {
                                ?>

                                <a href="<?php echo get_edit_user_link( $user->ID ); ?>"><?php echo $user->display_name . ' (' . $user->user_login . ')'; ?></a>

                                <?php
                            } else {
                                echo $user->display_name;
                            }
                        } else {
                            echo __( 'No user assigned', 'automatorwp-pro' );
                        } ?></span>
                </div>

                <div class="misc-pub-section curtime misc-pub-curtime">
                    <span id="timestamp"><?php echo __( 'Date:', 'automatorwp-pro' ); ?> <b><abbr title="<?php echo date( 'Y/m/d g:i:s a', strtotime( $object->date ) ); ?>"><?php echo date( 'Y/m/d', strtotime( $object->date ) ); ?></abbr></b></span>
                </div>

            </div>

        </div>

        <div id="major-publishing-actions">

            <div id="delete-action">
                <?php
                printf(
                    '<a href="%s" class="submitdelete deletion" onclick="%s" aria-label="%s">%s</a>',
                    ct_get_delete_link( $ct_table->name, $object->id ),
                    "return confirm('" .
                    esc_attr( __( "Are you sure you want to delete this item?\\n\\nClick \\'Cancel\\' to go back, \\'OK\\' to confirm the delete." ) ) .
                    "');",
                    esc_attr( __( 'Delete permanently' ) ),
                    __( 'Delete Permanently' )
                );
                ?>
            </div>

            <div class="clear"></div>

        </div>

    </div>
    <?php
}

/**
 * Render the log data meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_webhooks_log_data_meta_box( $object = null ) {

    $log_fields = automatorwp_webhooks_get_log_fields( $object );

    // Setup the CMB2 form
    $cmb2 = new CMB2( array(
        'id'        => 'automatorwp_webhooks_logs_form',
        'object_types' => array( 'automatorwp_webhooks_logs' ),
        'classes'   => 'automatorwp-form automatorwp-log-form',
        'hookup'    => false,
    ), $object->id );

    // Setup the options fields
    foreach ( $log_fields as $field_id => $field ) {

        $field['id'] = $field_id;

        if( $field['type'] !== 'title' && ! isset( $field['before_field'] ) ) {
            $field['before_field'] = 'automatorwp_webhooks_log_field_cb';
        }

        if( ! isset( $field['attributes'] ) ) {
            $field['attributes'] = array();
        }

        $field['attributes']['type'] = 'hidden';

        // Add the field to the form
        $cmb2->add_field( $field );

    }

    // Render the form
    CMB2_Hookup::enqueue_cmb_css();
    CMB2_Hookup::enqueue_cmb_js();
    $cmb2->show_form();

    ?>

    <?php
}

/**
 * Get log fields
 *
 * @since 1.0.0
 *
 * @param stdClass $log The log object
 *
 * @return array
 */
function automatorwp_webhooks_get_log_fields( $log ) {

    // Setup the log field
    $log_fields = array(
        'url' => array(
            'name' => __( 'URL:', 'automatorwp-pro' ),
            'desc' => __( 'URL requested.', 'automatorwp-pro' ),
            'type' => 'text',
        ),
        'method' => array(
            'name' => __( 'Method:', 'automatorwp-pro' ),
            'desc' => __( 'Request HTTP method.', 'automatorwp-pro' ),
            'type' => 'text',
        ),
        'data' => array(
            'name' => __( 'Data:', 'automatorwp-pro' ),
            'desc' => __( 'Data received.', 'automatorwp-pro' ),
            'type' => 'text',
        )
    );

    /**
     * Filter to set custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     *
     * @return array
     */
    return apply_filters( 'automatorwp_webhooks_log_fields', $log_fields, $log );

}

/**
 * Callback used to render action on logs
 *
 * @since 1.0.0
 *
 * @param array         $field_args
 * @param CMB2_Field    $field
 */
function automatorwp_webhooks_log_field_cb( $field_args, $field ) {

    global $ct_table;

    // Setup vars
    $primary_key = $ct_table->db->primary_key;
    $log_id = $_GET[$primary_key];
    $log = ct_get_object( $log_id );
    $field_id = $field_args['id'];
    $value = $field->value();

    $value = maybe_unserialize( $value );

    /**
     * Filters the field value display
     *
     * @since 1.0.0
     *
     * @param string        $value      Field value
     * @param array         $field_args Field parameters
     * @param CMB2_Field    $field      Field object
     * @param stdClass      $log        Log object
     *
     * @return string
     */
    $value = apply_filters( "automatorwp_webhooks_log_{$field_id}_field_value_display", $value, $field_args, $field, $log );

    // if empty value, set it as a space
    if( empty( $value ) ) {
        $value = '&nbsp;';
    }

    // Check if value is an array
    if( is_array( $value ) ) {

        $value = automatorwp_log_array_display( $value );

    }

    echo $value;

}