<?php
/**
 * User Lists Users
 *
 * @package     AutomatorWP\User_Lists\Custom_Tables\User_Lists_Users
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Parse query args for fields
 *
 * @since 1.0.0
 *
 * @param string $where
 * @param CT_Query $ct_query
 *
 * @return string
 */
function automatorwp_user_lists_users_query_where( $where, $ct_query ) {

    global $ct_table;

    if( $ct_table->name !== 'automatorwp_user_lists_users' ) {
        return $where;
    }

    // Shorthand
    $qv = $ct_query->query_vars;

    // List ID
    $where .= automatorwp_custom_table_where( $qv, 'list_id', 'integer' );

    // User ID
    $where .= automatorwp_custom_table_where( $qv, 'user_id', 'integer' );

    return $where;
}
add_filter( 'ct_query_where', 'automatorwp_user_lists_users_query_where', 10, 2 );

/**
 * Define the search fields for custom fields
 *
 * @since 1.0.0
 *
 * @param array $search_fields
 *
 * @return array
 */
function automatorwp_user_lists_users_search_fields( $search_fields ) {

    $search_fields[] = 'list_id';
    $search_fields[] = 'user_id';

    return $search_fields;

}
add_filter( 'ct_query_automatorwp_user_lists_users_search_fields', 'automatorwp_user_lists_users_search_fields' );

/**
 * Columns for custom fields list view
 *
 * @since 1.0.0
 *
 * @param array $columns
 *
 * @return array
 */
function automatorwp_user_lists_users_manage_lists_columns( $columns = array() ) {

    $columns['user_id']         = __( 'User', 'automatorwp-pro' );
    $columns['email']           = __( 'Email', 'automatorwp-pro' );
    $columns['actions']         = __( 'Actions', 'automatorwp-pro' );

    return $columns;
}
add_filter( 'manage_automatorwp_user_lists_users_columns', 'automatorwp_user_lists_users_manage_lists_columns' );

/**
 * Sortable columns for custom fields list view
 *
 * @since 1.0.0
 *
 * @param array $sortable_columns
 *
 * @return array
 */
function automatorwp_user_lists_users_manage_lists_sortable_columns( $sortable_columns ) {

    $sortable_columns['user_id']               = array( 'user_id', false );

    return $sortable_columns;

}
add_filter( 'manage_automatorwp_user_lists_users_sortable_columns', 'automatorwp_user_lists_users_manage_lists_sortable_columns' );

/**
 * Parse search query args for user earnings
 *
 * @since 1.6.4.2
 *
 * @param string $search
 * @param CT_Query $ct_query
 *
 * @return string
 */
function automatorwp_user_lists_users_query_search( $search, $ct_query ) {

    global $ct_table, $wpdb;

    if( $ct_table->name !== 'automatorwp_user_lists_users' ) {
        return $search;
    }

    $table_name = $ct_table->db->table_name;

    // Shorthand
    $qv = $ct_query->query_vars;

    // Check if is search and query is not filtered by an specific user
    if( isset( $qv['s'] ) && ! empty( $qv['s'] ) && ! isset( $qv['user_id'] ) ) {

        // Made a user sub-search to retrieve them
        $users = get_users( array(
            'search' => sprintf( '*%s*', $qv['s'] ),
            'search_columns' => array(
                'user_login',
                'user_email',
                'display_name',
            ),
            'fields' => 'ID',
        ) );

        if( ! empty( $users ) ) {
            $search .= " OR ( {$table_name}.user_id IN (" . implode( ',', array_map( 'absint', $users ) ) . ") )";
        }

    }

    return $search;

}
add_filter( 'ct_query_search', 'automatorwp_user_lists_users_query_search', 10, 2 );

/**
 * Bulk actions for list view
 *
 * @since  1.3.9.7
 *
 * @param array $actions
 *
 * @return array
 */
function automatorwp_user_lists_users_custom_bulk_actions( $actions = array() ) {

    return array();

}
add_filter( 'automatorwp_user_lists_users_bulk_actions', 'automatorwp_user_lists_users_custom_bulk_actions' );

/**
 * Row actions for user earnings list view
 *
 * @since  1.3.9.7
 *
 * @param array $actions
 *
 * @return array
 */
function automatorwp_user_lists_users_custom_row_actions( $actions, $object ) {

    return array();

}
add_filter( 'automatorwp_user_lists_users_row_actions', 'automatorwp_user_lists_users_custom_row_actions', 10, 2 );

/**
 * Columns rendering for custom fields list view
 *
 * @since  1.0.0
 *
 * @param string $column_name
 * @param integer $object_id
 */
function automatorwp_user_lists_users_manage_list_custom_column(  $column_name, $object_id ) {

    // Setup vars
    $list_user = ct_get_object( $object_id );

    switch( $column_name ) {
        case 'user_id':
            $user = get_userdata( $list_user->user_id );

            if( $user ) {

                if( current_user_can( 'edit_users' ) ) {
                    ?>

                    <a href="<?php echo get_edit_user_link( $user->ID ); ?>"><?php echo $user->display_name . ' (' . $user->user_login . ')'; ?></a>

                    <?php
                } else {
                    echo $user->display_name;
                }

            } else {
                echo __( 'No user assigned', 'automatorwp-webhooks' );
            }
            break;
        case 'email':
            $user = get_userdata( $list_user->user_id );

            if( $user ) {

                echo $user->user_email;

            } else {
                echo '';
            }
            break;
        case 'actions':
            // Remove
            printf(
                '<a href="%s" class="submitdelete button automatorwp-button-danger automatorwp-user-lists-remove-button" aria-label="%s" title="%s" data-user-id="%d">%s</a>',
                ct_get_delete_link( 'automatorwp_user_lists_users', $list_user->id ),
                esc_attr( __( 'Remove from list', 'automatorwp-pro' ) ),
                esc_attr( __( 'Remove from list', 'automatorwp-pro' ) ),
                esc_attr($list_user->user_id),
                automatorwp_dashicon('trash')
            );
            break;
    }
}
add_action( 'manage_automatorwp_user_lists_users_custom_column', 'automatorwp_user_lists_users_manage_list_custom_column', 10, 2 );