<?php
/**
 * Order Status
 *
 * @package     AutomatorWP\Integrations\Upsell_Plugin\Triggers\Order_Status
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Upsell_Plugin_Order_Status extends AutomatorWP_Integration_Trigger {

    public $integration = 'upsell_plugin';
    public $trigger = 'upsell_plugin_order_status';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User\'s order status changes', 'automatorwp-pro' ),
            'select_option'     => __( 'User\'s <strong>order status</strong> changes', 'automatorwp-pro' ),
            /* translators: %1$s: Order status. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User\'s order status changes to %1$s %2$s time(s)', 'automatorwp-pro' ), '{status}', '{times}' ),
            /* translators: %1$s: Order status. */
            'log_label'         => sprintf( __( 'User\'s order status changes to %1$s', 'automatorwp-pro' ), '{status}' ),
            'action'            => 'upsell_order_status_changed',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'status' => array(
                    'from' => 'status',
                    'fields' => array(
                        'status' => array(
                            'name' => __( 'Status:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'pending' => __( 'Pending', 'automatorwp-pro' ),
                                'active' => __( 'Active', 'automatorwp-pro' ),
                                'failed' => __( 'Failed', 'automatorwp-pro' ),
                                'completed' => __( 'Completed', 'automatorwp-pro' ),
                                'cancelled' => __( 'Cancelled', 'automatorwp-pro' ),
                                'refunded' => __( 'Refunded', 'automatorwp-pro' ),
                                'expired' => __( 'Expired', 'automatorwp-pro' ),
                            ),
                            'default' => 'pending'
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param string                $status_from
     * @param string                $status_to
     * @param Upsell\Entities\Order $order Order object
     */
    public function listener( $status_from, $status_to, $order ) {

        // Bail if status hasn't been changed
        if( $status_from === $status_to ) {
            return;
        }

        $order_id = $order->getId();
        $user_id = $order->customer()->attribute('user_id');

        // Trigger the order status change
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'order_id'      => $order_id,
            'status'        => $status_to,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if status is not received
        if( ! isset( $event['status'] ) ) {
            return false;
        }

        // Don't deserve if status doesn't matches with the trigger option
        if( $event['status'] !== $trigger_options['status'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['order_id'] = ( isset( $event['order_id'] ) ? $event['order_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_Upsell_Plugin_Order_Status();