<?php
/**
 * Complete Quiz Percentage Category
 *
 * @package     AutomatorWP\Integrations\Tutor_LMS\Triggers\Complete_Quiz_Percentage_Category
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Tutor_LMS_Complete_Quiz_Percentage_Category extends AutomatorWP_Integration_Trigger {

    public $integration = 'tutor';
    public $trigger = 'tutor_complete_quiz_percentage_category';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a quiz of a category with a grade percentage greater than, less than or equal to a specific percentage', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a quiz of a category with a grade percentage <strong>greater than, less than or equal</strong> to a specific percentage', 'automatorwp-pro' ),
            /* translators: %1$s: Category. %2$s: Condition. %3$s: Percent. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes a quiz of %1$s category with a grade percentage %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{term}', '{condition}', '{percentage}', '{times}' ),
            /* translators: %1$s: Category. %2$s: Condition. %3$s: Percent. */
            'log_label'         => sprintf( __( 'User completes a quiz of %1$s category with a grade percentage %2$s %3$s', 'automatorwp-pro' ), '{term}', '{condition}', '{percentage}' ),
            'action'            => 'tutor_quiz/attempt_ended',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'taxonomy' => 'course-category',
                )),
                'condition' => automatorwp_utilities_condition_option(),
                'percentage' => array(
                    'from' => 'percentage',
                    'fields' => array(
                        'percentage' => array(
                            'name' => __( 'Percentage:', 'automatorwp-pro' ),
                            'desc' => __( 'The grade percentage required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $attempt_id
     */
    public function listener( $attempt_id ) {

        $attempt = tutor_utils()->get_attempt( $attempt_id );
        $user_id = get_current_user_id();
        $quiz_id = $attempt->quiz_id;

        // Bail if is not a quiz
        if ( 'tutor_quiz' !== get_post_type( $quiz_id ) ) {
            return;
        }

        // Bail if attempt isn't finished yet
        if ( ! in_array( $attempt->attempt_status, array( 'attempt_ended', 'review_required' ) ) ) {
            return;
        }

        $course = tutor_utils()->get_course_by_quiz( $quiz_id );
        $course_id = $course->ID;
        $percentage = ( $attempt->earned_marks * 100 ) / $attempt->total_marks;

        $terms_ids = automatorwp_get_term_ids( $course_id, 'course-category' );

        // Bail if post isn't assigned to any category
        if( empty( $terms_ids ) ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $quiz_id,
            'course_id'     => $course_id,
            'percentage'    => $percentage,
            'terms_ids'     => $terms_ids,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['percentage'] ) ) {
            return false;
        }

        // Don't deserve if category is not received
        if( ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        $percentage = (float) $event['percentage'];
        $required_percentage =  (float) $trigger_options['percentage'];

        // Don't deserve if order total doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $percentage, $required_percentage, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Tutor_LMS_Complete_Quiz_Percentage_Category();