<?php
/**
 * User Is Not Enrolled Course
 *
 * @package     AutomatorWP\Integrations\Tutor_LMS\Filters\User_Is_Not_Enrolled_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Tutor_LMS_User_Is_Not_Enrolled_Course_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'tutor';
    public $filter = 'tutor_user_is_not_enrolled_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is not enrolled in course', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>is not enrolled</strong> in course', 'automatorwp-pro' ),
            /* translators: %1$s: Course. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}'  ),
            /* translators: %1$s: Course. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Course:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any course', 'automatorwp-pro' ),
                    'post_type' => apply_filters( 'tutor_course_post_type', 'courses' ),
                    'default' => 'any'
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $course = $filter_options['post'];

        if ($course !== 'any') {
            $course_name = get_post($course)->post_title;
        } else{
            $course_name = 'any course';
        }

        // Bail if wrong configured
        if( empty( $course ) ) {
            $this->result = __( 'Filter not passed. Course option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        // Get courses where the student is enrolled
        $courses = tutor_utils()->get_enrolled_courses_ids_by_user( $user_id );

        // Don't deserve if user is in this course
        if( in_array( $course, $courses ) && ! empty( $courses ) ) {
            $this->result = sprintf( __( 'Filter not passed. User is enrolled in course "%1$s".', 'automatorwp-pro' ),
                $course_name
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User is not enrolled in course "%1$s".', 'automatorwp-pro' ),
            $course_name
        );

        return $deserves_filter;

    }

}

new AutomatorWP_Tutor_LMS_User_Is_Not_Enrolled_Course_Filter();