<?php
/**
 * Purchase Ticket Category
 *
 * @package     AutomatorWP\Integrations\The_Events_Calendar\Triggers\Purchase_Ticket_Category
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_The_Events_Calendar_Purchase_Ticket_Category extends AutomatorWP_Integration_Trigger {

    public $integration = 'the_events_calendar';
    public $trigger = 'the_events_calendar_purchase_ticket_category';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User purchases a ticket for an event in a category', 'automatorwp-pro' ),
            'select_option'     => __( 'User purchases <strong>a ticket</strong> for an event in a category', 'automatorwp-pro' ),
            /* translators: %1$s: Category. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User purchases a ticket for an event in %1$s %2$s time(s)', 'automatorwp-pro' ), '{term}', '{times}' ),
            /* translators: %1$s: Category. */
            'log_label'         => sprintf( __( 'User purchases a ticket for an event in %1$s', 'automatorwp-pro' ), '{term}' ),
            'action'            => 'tec_tickets_commerce_order_status_flag_complete',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'term' => automatorwp_utilities_term_option( array(
                    'taxonomy' => 'tribe_events_cat',
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param TECTicketsCommerceStatusStatus_Interface      $new_status     New post status
     * @param TECTicketsCommerceStatusStatus_Interface|null $old_status     Old post status
     * @param WP_Post   $post   Post object
     */
    public function listener( $new_status, $old_status, $post ) {

        $user_id = get_current_user_id();
        $order_data = tec_tc_get_order( $post );

        // Bail if order is not completed
        if ( $order_data->post_status !== 'tec-tc-completed' ) {
            return;
        }

        foreach( $order_data->items as $ticket_id => $value ) {
            $event_id = $value['event_id'];

            $terms_ids = automatorwp_get_term_ids( $event_id, 'tribe_events_cat' );

            // Skip items that aren't assigned to any category
            if( empty( $terms_ids ) ) {
                continue;
            }

            for ( $i = absint( $value['quantity'] ); $i > 0; $i -= 1 ){

                // Trigger the ticket purchase
                automatorwp_trigger_event( array(
                    'trigger'       => $this->trigger,
                    'user_id'       => $user_id,
                    'post_id'       => $event_id,
                    'ticket_id'     => $ticket_id,
                    'terms_ids'     => $terms_ids,
                ) );
            }
            
        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post and terms IDs are not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['terms_ids'] ) ) {
            return false;
        }

        // Don't deserve if term doesn't match with the trigger option
        if( ! automatorwp_terms_matches( $event['terms_ids'], $trigger_options['term'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_The_Events_Calendar_Purchase_Ticket_Category();