<?php
/**
 * Check-in Event
 *
 * @package     AutomatorWP\Integrations\The_Events_Calendar\Triggers\Checkin_Event
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_The_Events_Calendar_Checkin_Event extends AutomatorWP_Integration_Trigger {

    public $integration = 'the_events_calendar';
    public $trigger = 'the_events_calendar_checkin_event';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User check-ins at event', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>check-ins</strong> at event', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User check-ins at %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'User check-ins at %1$s', 'automatorwp-pro' ), '{post}' ),
            'action'            => array(
                'event_tickets_checkin',
                'rsvp_checkin',
                'wootickets_checkin',
                'eddtickets_checkin',
            ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Event:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any event', 'automatorwp-pro' ),
                    'post_type' => Tribe__Events__Main::POSTTYPE
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $attendee_id    Attendee post ID
     * @param bool|null $qr             true if from QR checkin process
     */
    public function listener( $attendee_id, $qr ) {

        $attendees = tribe_tickets_get_attendees( $attendee_id, 'rsvp_order' );

        // Bail if can not find the attendees
        if ( empty( $attendees ) ) {
            return;
        }

        foreach ( $attendees as $attendee ) {

            $user_id  = absint( $attendee['user_id'] );
            $event_id = absint( $attendee['event_id'] );

            // If user was created after performing the RSVP
            if ( $user_id === 0 ) {
                $user = get_user_by( 'email', $attendee['purchaser_email'] );
                
                if ( $user )
                    $user_id = $user->ID;
            }

            // Trigger the event check-in
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'post_id'       => $event_id,
                'qr'            => $qr,
            ) );

        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_The_Events_Calendar_Checkin_Event();