<?php
/**
 * Status Membership
 *
 * @package     AutomatorWP\Integrations\Restrict_Content_Pro\Triggers\Status_Membership
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Restrict_Content_Pro_Status_Membership extends AutomatorWP_Integration_Trigger {

    public $integration = 'restrict_content_pro';
    public $trigger = 'restrict_content_pro_status_membership';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets status in membership manually', 'automatorwp-pro' ),
            'select_option'     => __( 'User gets <strong>status</strong> in membership manually', 'automatorwp-pro' ),
            /* translators: %1$s: Status. %2$s: Membership. %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets %1$s in %2$s %3$s time(s)', 'automatorwp-pro' ), '{status}', '{membership}', '{times}' ),
            /* translators: %1$s: Status. %2$s: Membership. */
            'log_label'         => sprintf( __( 'User gets %1$s in %2$s', 'automatorwp-pro' ), '{status}', '{membership}' ),
            'action'            => 'rcp_after_membership_admin_update',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'membership' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'membership',
                    'name'              => __( 'Membership Level:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any membership level', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_restrict_content_pro_get_memberships',
                    'options_cb'        => 'automatorwp_restrict_content_pro_options_cb_membership',
                    'default'           => 'any'
                ) ),
                'status' => array(
                    'from' => 'status',
                    'default' => __( 'any status', 'automatorwp-pro' ),
                    'fields' => array(
                        'status' => array(
                            'name'          => __( 'Status:', 'automatorwp-pro' ),
                            'type'          => 'select',
                            'options_cb'    => 'automatorwp_restrict_content_pro_options_cb_status',
                            'classes'       => 'automatorwp-selector',
                            'default'       => 'any'
                    
                 ))),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_restrict_content_pro_get_membership_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param RCP_Membership $membership    Membership object.
     * @param array          $args
     */
    public function listener(  $membership, $args ) {

        // Bail if no status modified
        if ( !isset( $args['status'] ) ) {
            return;
        }
        
        $user_id     = $membership->get_user_id();
        $level_id = $membership->get_object_id();

        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user_id,
            'level_id'          => $level_id,
            'membership_id'     => $membership->get_id(),
            'status'            => $args['status'],
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if membership is not received
        if( ! isset( $event['level_id'] ) ) {
            return false;
        }

        // Don't deserve if status is not received
        if( ! isset( $event['status'] ) ) {
            return false;
        }

        $level_id = absint( $event['level_id'] );

        // Don't deserve if membership doesn't exists
        if( $level_id === 0 ) {
            return false;
        }

        $required_level_id = absint( $trigger_options['membership'] );

        // Don't deserve if membership doesn't match with the trigger option
        if( $trigger_options['membership'] !== 'any' && $level_id !== $level_id ) {
            return false;
        }

        $status = $event['status'];
        $required_status = $trigger_options['status'];

        // Don't deserve if status doesn't match with the trigger option
        if( $trigger_options['status'] !== 'any' && $status !== $required_status ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['membership_id'] = ( isset( $event['membership_id'] ) ? $event['membership_id'] : '' );

        return $log_meta;

    }

}

new AutomatorWP_Restrict_Content_Pro_Status_Membership();