<?php
/**
 * Add User Notification
 *
 * @package     AutomatorWP\Integrations\PeepSo\Actions\Add_User_Notification
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_PeepSo_Add_User_Notification extends AutomatorWP_Integration_Action {

    public $integration = 'peepso';
    public $action = 'peepso_add_user_notification';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add a notification to the user', 'automatorwp-pro' ),
            'select_option'     => __( 'Add a <strong>notification</strong> to the user', 'automatorwp-pro' ),
            /* translators: %1$s: Notification. */
            'edit_label'        => sprintf( __( 'Add a %1$s to the user', 'automatorwp-pro' ), '{notification}' ),
            /* translators: %1$s: Notification. */
            'log_label'         => sprintf( __( 'Add a %1$s to the user', 'automatorwp-pro' ), '{notification}' ),
            'options'           => array(
                'notification' => array(
                    'default' => __( 'notification', 'automatorwp-pro' ),
                    'fields' => array(
                        'from' => array(
                            'name' => __( 'From:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that sends the notification. Leave blank to set the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'to' => array(
                            'name' => __( 'To:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that will receive the notification. Leave blank to send the notification to the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Notification\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Bail if notifications component is not active
        if ( ! class_exists( 'PeepSoNotifications' ) ) {
            $this->result = __( 'Class PeepSoNotifications not found.', 'automatorwp-pro' );
            return;
        }

        // Setup the user who will receive the notification
        $from = $action_options['from'];
        $to = $action_options['to'];

        // Setup from
        if( empty( $from ) ) {
            $from = $user_id;
        }

        if( absint( $from ) === 0 ) {
            $this->result = __( 'Invalid "from" user provided in the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        if( empty( $to ) ) {
            $to = $user_id;
        }

        if( absint( $to ) === 0 ) {
            $this->result = __( 'Invalid "to" user provided in the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        $arg = array ( "peepso-core");

        // To create the activity for the notification
        $PeepSoActivity = new PeepSoActivity();
		 
        $result_activity = $PeepSoActivity->add_post($to, $from, $action_options['content']);

        // Create Notification
        if ($from === $to ) {
            $PeepSoNotifications = new PeepSoNotifications();
        
            $result = $PeepSoNotifications->add_notification_new( $from, $to, $action_options['content'], $arg, 'automatorwp', 1000, $result_activity, 0 );
        }
        
        if( $result ) {
            $this->result = __( 'User notified successfully.', 'automatorwp-pro' );
        } else {
            $this->result = __( 'User not notified.', 'automatorwp-pro' );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_PeepSo_Add_User_Notification();