<?php
/**
 * Create and Send Campaign
 *
 * @package     AutomatorWP\Integrations\Mailchimp\Actions\Create_Send_Campaign
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Mailchimp_Create_Send_Campaign extends AutomatorWP_Integration_Action {

    public $integration = 'mailchimp';
    public $action = 'mailchimp_create_send_campaign';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Create and send a campaign', 'automatorwp-pro' ),
            'select_option'     => __( 'Create and send a <strong>campaign</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Campaign. */
            'edit_label'        => sprintf( __( 'Create and send %1$s', 'automatorwp-pro' ), '{campaign}' ),
            /* translators: %1$s: Campaign. */
            'log_label'         => sprintf( __( 'Create and send %1$s', 'automatorwp-pro' ), '{campaign}' ),
            'options'           => array(
                'campaign' => array(
                    'from' => 'name',
                    'default' => __( 'campaign', 'automatorwp-pro' ),
                    'fields' => array(
                        'name' => array(
                            'name'  => __( 'Campaign name', 'automatorwp-pro' ),
                            'type'  => 'text',
                            'required'   => true,
                        ),
                        'audience' => automatorwp_utilities_ajax_selector_field( array(
                            'option_none' => false,
                            'option_custom' => false,
                            'placeholder'       => __( 'Select an audience', 'automatorwp-pro' ),
                            'name'              => __( 'Audience', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_mailchimp_get_lists',
                            'options_cb'        => 'automatorwp_mailchimp_options_cb_lists',
                            'required'          => true
                        ) ),
                        'tags' => automatorwp_utilities_ajax_selector_field( array(
                            'option_none' => false,
                            'option_custom' => false,
                            'placeholder'       => __( 'Select a tag or segment', 'automatorwp-pro' ),
                            'name'              => __( 'Tag or Segment', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_mailchimp_get_tags_and_segments',
                            'options_cb'        => 'automatorwp_mailchimp_options_cb_tags_and_segments'
                        ) ),
                        'subject_line' => array(
                            'name'  => __( 'Subject', 'automatorwp-pro' ),
                            'type'  => 'text',
                            'required' => true
                        ),
                        'preview_text' => array(
                            'name'  => __( 'Preview text', 'automatorwp-pro' ),
                            'type'  => 'text'
                        ),
                        'from_name' => array(
                            'name'  => __( 'From name', 'automatorwp-pro' ),
                            'type'  => 'text',
                            'required' => true
                        ),
                        'reply_to' => array(
                            'name'  => __( 'From email', 'automatorwp-pro' ),
                            'type'  => 'text',
                            'required' => true
                        ),
                        'to_name' => array(
                            'name'  => __( 'Merge tags', 'automatorwp-pro' ),
                            'description' => __( 'E.g: *|FNAME|*, *|LNAME|*. You will be able to use this tags on your content', 'automatorwp-pro' ),
                            'type'  => 'text'
                        ),
                        'template_id' => automatorwp_utilities_ajax_selector_field( array(
                            'option_none' => false,
                            'option_custom' => false,
                            'placeholder'       => __( 'Select a template', 'automatorwp-pro' ),
                            'name'              => __( 'Template', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_mailchimp_get_templates',
                            'options_cb'        => 'automatorwp_mailchimp_options_cb_templates'
                        ) ),
                        'content' => array(
                            'name'  => __( 'Campaign content', 'automatorwp-pro' ),
                            'desc'  => __( 'Email content for the campaign to send', 'automator-mailchimp'),
                            'type'  => 'wysiwyg',

                        ),
                                                
                    ),
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Shorthand
        $user = get_user_by ( 'ID', $user_id );
        $content = wpautop( $action_options['content'] );
        $list_id = $action_options['audience'];
        $tag_id = $action_options['tags'];
        $template_id = $action_options['template_id'];

        if ( $list_id === 'any' ){
			$this->result = __( 'Unselected Audience ', 'automatorwp-pro' );
			return;
		}

        // Campaign general settings
        $settings = (object) array(
            'subject_line'      => $action_options['subject_line'],
            'preview_text'      => $action_options['preview_text'],
            'title'             => $action_options['name'],
            'from_name'         => $action_options['from_name'],
            'to_name'           => $action_options['to_name'],
            'reply_to'          => $action_options['reply_to']
        );

        // Check if template was selected
        if ( $template_id == 'any' ){

            $content_type = 'multichannel';
            $campaign_content = array(
                'html' => $content,
            );
    
        } else{

            $content_type = 'template';
            $settings->template_id = (int) $template_id;

        }
        
        // Check if tag was selected
        if ( $tag_id == 'any' ){

            // To send the campaig to all users
            $recipients = (object) array(
                'list_id' => $list_id
            );
    
        }else{

            // To send the campaign only to users with the selected tag
            $recipients = (object) array(
                'list_id' => $list_id,
                'segment_opts' => (object) array(
                    'saved_segment_id' => (int) $tag_id // INTEGER
                )
            );

        }

        // Get object to connect to API
        $mailchimp = automatorwp_mailchimp_get_authorization();

        // Bail if no authorization
        if ( ! $mailchimp ){
            return;
        }

        try {

            // To create the new campaign
            $response = $mailchimp->campaigns->create(['type' => 'regular', 'recipients' => $recipients, 'settings' => $settings, 'content-type' => $content_type ]);
            $campaign_id = $response->id;
            
            // To put the content for the new campaign
            if ( $template_id == 'any' ){
                $response = $mailchimp->campaigns->setContent($campaign_id, ['html' => $content]);
            }

            // To send the new campaign
            $campaign_sent = $mailchimp->campaigns->send( $campaign_id );
        
            $this->result = __( 'Campaign sent', 'automatorwp-pro' );


        } catch (\GuzzleHttp\Exception\ClientException $e) {

            $response = $e->getResponse();
            $responseBodyAsString = $response->getBody()->getContents();
            $decoded = json_decode($responseBodyAsString);
            $this->result = $decoded->title;
            return;

        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Warn user if the authorization has not been setup from settings
        if( ! automatorwp_mailchimp_get_authorization() ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Mailchimp settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-mailchimp'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/mailchimp/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Mailchimp_Create_Send_Campaign();