<?php
/**
 * Click Link
 *
 * @package     AutomatorWP\Integrations\AutomatorWP\Triggers\Click_Link
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_AutomatorWP_Click_Link extends AutomatorWP_Integration_Trigger {

    /**
     * Initialize the action
     *
     * @since 1.0.0
     */
    public function __construct( $integration ) {

        $this->integration = $integration;
        $this->trigger = $integration . '_click_link';

        parent::__construct();

    }

    public function hooks() {

        // Custom trigger replacements
        add_filter( 'automatorwp_parse_automation_item_label_replacements', array( $this, 'item_label_replacements' ), 10, 5 );

        // Custom html on automations UI
        add_action( 'automatorwp_automation_ui_after_item_label', array( $this, 'after_item_label' ) );

        // Custom title replacement
        add_filter( 'automatorwp_clone_trigger_automatorwp_click_link_title', array( $this, 'item_label_clone_replacements' ), 10, 3 );

        parent::hooks();
    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User clicks on a link', 'automatorwp-pro' ),
            'select_option'     => __( 'User clicks on <strong>a link</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Link shortcode. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User clicks on %1$s %2$s time(s)', 'automatorwp-pro' ), '{link}', '{times}' ),
            /* translators: %1$s: Link shortcode. */
            'log_label'         => sprintf( __( 'User clicks on %1$s', 'automatorwp-pro' ), '{link}' ),
            'action'            => 'automatorwp_link_clicked',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 6,
            'options'           => array(
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Parses item title replacements
     *
     * @since 1.0.0
     *
     * @param string    $item_title      The item title
     * @param int       $old_item_id     The old item ID
     * @param int       $new_item_id     The new item ID
     *
     * @return array
     */
    public function item_label_clone_replacements( $item_title, $old_item_id, $new_item_id ) {
    
        // ID replacement
        $item_title = str_replace($old_item_id, $new_item_id, $item_title);
    
        return $item_title;
        
    }
    
    /**
     * Parses trigger label replacements
     *
     * @since 1.0.0
     *
     * @param array     $replacements   The replacements to apply
     * @param stdClass  $object         The trigger object
     * @param string    $item_type      The item type (trigger|action)
     * @param string    $label          The label to parse
     * @param string    $context        The context this function is executed
     *
     * @return array
     */
    public function item_label_replacements( $replacements, $object, $item_type, $label, $context ) {

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $replacements;
        }

        // Add the link replacement
        $replacements['{link}'] = '[automatorwp_link trigger="' . $object->id . '"]';

        // On edit context, prevent to grab
        if( $context === 'edit' ) {
            $replacements['{link}'] = '<span class="automatorwp-no-grab">' . $replacements['{link}'] . '</span>';
        }

        return $replacements;

    }

    /**
     * Custom html on automations UI
     *
     * @since 1.0.0
     *
     * @param stdClass $object The trigger/action object
     */
    public function after_item_label( $object ) {

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return;
        }

        $attributes = array(
            'trigger'   => __( 'The trigger identifier. Leave the same value as provided by the trigger.', 'automatorwp-pro' ),
            'label'     => __( 'The link label text. By default: "Click me!"', 'automatorwp-pro' ),
            'href'      => __( 'The link url.', 'automatorwp-pro' ),
            'id'        => __( 'The link id attribute.', 'automatorwp-pro' ),
            'class'     => __( 'The link class attribute.', 'automatorwp-pro' ),
            'target'    => __( 'The link target attribute. The target attribute specifies where to open the linked page', 'automatorwp-pro' ),
            'title'     => __( 'The link title attribute.', 'automatorwp-pro' ),
        );

        ?>
        <a href="#" class="automatorwp-no-grab" style="margin-top: 5px; cursor: pointer;"
           onclick="jQuery(this).next().slideToggle('fast'); return false;"
        ><?php _e( 'Shortcode attributes', 'automatorwp-pro' ); ?></a>
        <div class="automatorwp-link-shortcode-attributes automatorwp-no-grab" style="display: none;">
            <ul>
                <?php foreach( $attributes as $attr => $desc ) : ?>
                    <li>
                        <strong><?php echo $attr; ?></strong> - <span style="color: #757575;"><?php echo $desc; ?></span>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>

        <?php

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $trigger_id     The trigger ID
     * @param int       $user_id        The user ID
     * @param int       $post_id        The post ID
     * @param string    $url            The link url attribute
     * @param string    $id             The link id attribute
     * @param string    $class          The link class attribute
     */
    public function listener( $trigger_id, $user_id, $post_id, $url, $id, $class ) {
        
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'trigger_id'    => $trigger_id,
            'post_id'       => $post_id,
            'link_url'      => $url,
            'link_id'       => $id,
            'link_class'    => $class,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if automation is not received
        if( ! isset( $event['trigger_id'] ) ) {
            return false;
        }

        $trigger_id = absint( $event['trigger_id'] );

        // Don't deserve if trigger doesn't match with the trigger option
        if( absint( $trigger->id ) !== $trigger_id ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_AutomatorWP_Click_Link( 'automatorwp' );
new AutomatorWP_AutomatorWP_Click_Link( 'link' );