<?php
/**
 * User Is Not Added Group
 *
 * @package     AutomatorWP\Integrations\LifterLMS\Filters\User_Is_Not_Added_Group
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LifterLMS_User_Is_Not_Added_Group_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'lifterlms';
    public $filter = 'lifterlms_user_is_not_added_group';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is not in group', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>is not in group</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}'  ),
            /* translators: %1$s: Group. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Group:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any group', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'post_type' => 'llms_group',
                    'default' => 'any'
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $group = $filter_options['post'];

        if ( $group !== 'any' ) {
            $group_name = get_post($group)->post_title;
        } else{
            // Get all group IDs
            $all_groups = get_posts( array (
                'post_type'     => 'llms_group',
                'post_status'   => array( 'publish' ),
                'fields'        => 'ids',
            ) );

            $group_name = 'any group';
        }
        
        // Bail if wrong configured
        if( empty( $group ) ) {
            $this->result = __( 'Filter not passed. Group option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        if ( $group !== 'any' ) {
            // Check if user is member
            $is_member = LLMS_Groups_Enrollment::get_role( $user_id, $group );
        } else {
            // check if user is member in any group
            foreach( $all_groups as $group_id ) {
                $is_member = LLMS_Groups_Enrollment::get_role( $user_id, $group_id );
                if ( empty( $is_member ) ) {
                    break;
                }
            }
        }
        
        // Don't deserve if user is in this group
        if( !empty( $is_member ) ) {
            $this->result = sprintf( __( 'Filter not passed. User is in group "%1$s".', 'automatorwp-pro' ),
                $group_name
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User is not in group "%1$s".', 'automatorwp-pro' ),
            $group_name
        );

        return $deserves_filter;

    }

}

new AutomatorWP_LifterLMS_User_Is_Not_Added_Group_Filter();