<?php
/**
 * User Course Extend Access
 *
 * @package     AutomatorWP\Integrations\LearnDash\Actions\User_Course_Extend_Access
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_User_Course_Extend_Access extends AutomatorWP_Integration_Action {

    public $integration = 'learndash';
    public $action = 'learndash_user_course_extend_access';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Extend user access to a course by a number of days', 'automatorwp-pro' ),
            'select_option'     => __( '<strong>Extend</strong> user access to a course by a number of days', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Operation (completed or not completed). */
            'edit_label'        => sprintf( __( 'Extend user access to %1$s by %2$s days', 'automatorwp-pro' ), '{post}', '{days}' ),
            /* translators: %1$s: Post title. %2$s: Operation (completed or not completed). */
            'log_label'         => sprintf( __( 'Extend user access to %1$s by %2$s days', 'automatorwp-pro' ), '{post}', '{days}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_none_label'   => __( 'course', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type'         => 'sfwd-courses',
                    'default'   => __( 'course', 'automatorwp-pro' ),
                ) ),
                'days' => array(
                    'from' => 'days',
                    'fields' => array(
                        'days' => array(
                            'name' => __( 'Days:', 'automatorwp-pro' ),
                            'description' => __( 'The number of days to extend the course.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min'   => 1,
                            ),
                            'default' => 1,
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Check if function exists
		if ( ! function_exists( 'learndash_course_extend_user_access' ) ) {
            $this->result = __( 'Access to a course can be extended from LearnDash 4.8.0.', 'automatorwp-pro' );
			return;
		}

        // Shorthand
        $days = $action_options['days'];
        $course_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'complete';
        }

        // Check access to the course
		$has_access = sfwd_lms_has_access_fn( $course_id, $user_id );

        if ( !$has_access ) {
            $this->result = __( 'User is not enrolled in this course.', 'automatorwp-pro' );
            return;
        }

        // Check expire access
		$expire_access = learndash_get_setting( $course_id, 'expire_access' );

        if ( empty( $expire_access ) ) {
            $this->result = __( 'Course does not have expiration date.', 'automatorwp-pro' );
            return;
        }

        // Set new expiration date
		$expiration_date = ld_course_access_expires_on( $course_id, $user_id );
		$expiration_date = empty( $expiration_date ) ? time() : $expiration_date;
		$new_expiration_date = strtotime( '+' . $days . ' days', $expiration_date );
		
		learndash_course_extend_user_access( $course_id, array( $user_id ), $new_expiration_date );
       
        $this->result = sprintf (__( 'Extended access course until %s.', 'automatorwp-pro' ), date('Y-m-d', $new_expiration_date) );
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-activecampaign' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_LearnDash_User_Course_Extend_Access();