<?php
/**
 * Remove Attendee
 *
 * @package     AutomatorWP\Integrations\Google_Calendar\Actions\Remove_Attendee
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_Google_Calendar_Remove_Attendee extends AutomatorWP_Integration_Action {
    
    public $integration = 'google_calendar';
    public $action = 'google_calendar_remove_attendee';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Remove an attendee from an event', 'automatorwp-pro' ),
            'select_option' => __( 'Remove <strong>an attendee</strong> from an event', 'automatorwp-pro' ),
            /* translators: %1$s: Attendee. %2$s: Event. */
            'edit_label' => sprintf( __( 'Remove %1$s from %2$s', 'automatorwp-pro' ), '{attendee}', '{event}' ),
            /* translators: %1$s: Attendee. %2$s: Event. */
            'log_label' => sprintf( __( 'Remove %1$s from %2$s', 'automatorwp-pro' ), '{attendee}', '{event}' ),
            'options' => array(
                'event' => array(
                    'from' => 'event',
                    'default' => __( 'an event', 'automatorwp-pro' ),
                    'fields' => array(
                        'calendar' => automatorwp_utilities_ajax_selector_field( array(
                            'field'             => 'calendar',
                            'option_default'    => __( 'calendar', 'automatorwp-pro' ),
                            'placeholder'       => __( 'Select a calendar', 'automatorwp-pro' ),
                            'name'              => __( 'Calendar:', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_google_calendar_get_calendars',
                            'options_cb'        => 'automatorwp_google_calendar_options_cb_calendars',
                            'default'           => ''
                        ) ),
                        'event' => automatorwp_utilities_ajax_selector_field( array(
                            'field'             => 'event',
                            'option_default'    => __( 'event', 'automatorwp-pro' ),
                            'placeholder'       => __( 'Select an event', 'automatorwp-pro' ),
                            'name'              => __( 'Event:', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_google_calendar_get_events',
                            'options_cb'        => 'automatorwp_google_calendar_options_cb_events',
                            'default'           => ''
                        ) ),

                        )),
                        
                'attendee' => array(
                    'from' => 'attendee',
                    'default' => __( 'attendee', 'automatorwp-pro' ),
                    'fields' => array(
                        'attendee' => array(
                            'name' => __( 'Attendee:', 'automatorwp-pro' ),
                            'desc' => __( 'The attendee email.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                   
                    ),
                    )
                ),
          
        ));

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){

        // Bail if not selected calendar
        if( empty( $action_options['calendar'] ) ) {
            $this->result = __( 'No calendar selected on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Bail if not data to insert
        if( empty( $action_options['event'] ) ) {
            $this->result = __( 'No data found to insert.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $calendar_id = $action_options['calendar'];        
        $event_id = $action_options['event'];
        $attendee_to_delete = trim( $action_options['attendee'] );

        $params = automatorwp_google_calendar_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            return;
        }

        // Setup the request parameters
        $current_event = automatorwp_google_calendar_get_current_event( $calendar_id, $event_id );

        $current_event['attendees'] = array_filter( $current_event['attendees'], function( $attendee ) use ($attendee_to_delete) {
            return $attendee['email'] !== $attendee_to_delete;
        });

        $params['body'] = json_encode( $current_event );
        $params['method'] = 'PUT';

        // Query to add record
        
        $url = 'https://www.googleapis.com/calendar/v3/calendars/' . $calendar_id . '/events/' . $event_id;
      
        $request = wp_remote_request( $url, $params );

        $response = json_decode( wp_remote_retrieve_body( $request ), true );

        if ( isset( $response['error'] ) ) {

            if ( $response['error']['code'] === 400 ) {
                $this->result = sprintf( __( '%1$s', 'automatorwp-pro' ), $response['error']['message'] ); 
                return false;   
            }

        }

        $this->result = __( 'Attendee removed from the event successfully!', 'automatorwp-pro' );
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_Google_Calendar_Remove_Attendee();