<?php
/**
 * Donation
 *
 * @package     AutomatorWP\Integrations\Give\Triggers\Donation
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Give_Donation extends AutomatorWP_Integration_Trigger {

    public $integration = 'give';
    public $trigger = 'give_donation';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User performs a donation through a form', 'automatorwp-pro' ),
            'select_option'     => __( 'User performs a <strong>donation</strong> through a form', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User performs a donation through %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'User performs a donation through %1$s', 'automatorwp-pro' ), '{post}' ),
            'action'            => 'give_update_payment_status',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Form:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any from', 'automatorwp-pro' ),
                    'post_type' => 'give_forms'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_give_get_donation_tags(),
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $payment_id     The payment ID.
     * @param string    $new_status     The payment new status.
     * @param string    $old_status     The payment old status.
     */
    public function listener( $payment_id, $new_status, $old_status ) {

        // Make sure that payments are only completed once.
        if ( $old_status === 'publish' || $old_status === 'complete' ) {
            return;
        }

        // Make sure the payment completion is only processed when new status is complete.
        if ( $new_status !== 'publish' && $new_status !== 'complete' ) {
            return;
        }

        // Setup the payment data
        $payment        = new Give_Payment( $payment_id );
        $user_id        = $payment->user_id;
        $donor_id       = $payment->customer_id;
        $donor_email    = $payment->email;
        $total          = $payment->total;
        $form_id        = $payment->form_id;
        $form_title     = $payment->form_title;
        $donation_id    = $payment->ID;

        // Trigger the donation
        automatorwp_trigger_event( array(
            'trigger'               => $this->trigger,
            'user_id'               => $donor_id,
            'post_id'               => $form_id,
            'donation_total'        => $total,
            'donation_id'           => $donation_id,
            'donation_form_id'      => $form_id,
            'donation_form_title'   => $form_title,
            'donor_id'              => $donor_id,
            'donor_email'           => $donor_email,
        ) );

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['donation_id'] = ( isset( $event['donation_id'] ) ? $event['donation_id'] : '' );
        $log_meta['donation_total'] = ( isset( $event['donation_total'] ) ? $event['donation_total'] : '' );
        $log_meta['donor_id'] = ( isset( $event['donor_id'] ) ? $event['donor_id'] : '' );
        $log_meta['donor_email'] = ( isset( $event['donor_email'] ) ? $event['donor_email'] : '' );
        $log_meta['donation_form_id'] = ( isset( $event['donation_form_id'] ) ? $event['donation_form_id'] : '' );
        $log_meta['donation_form_title'] = ( isset( $event['donation_form_title'] ) ? $event['donation_form_title'] : '' );
        
        return $log_meta;

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Give_Donation();