<?php
/**
 * Generate Email
 *
 * @package     AutomatorWP\Integrations\Generator\Actions\Generate_Email
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Generator_Generate_Email extends AutomatorWP_Integration_Action {

    public $integration = 'generator';
    public $action = 'generator_generate_email';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Generate a random email', 'automatorwp-pro' ),
            'select_option'     => __( 'Generate a <strong>random email</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Random number. */
            'edit_label'        => sprintf( __( 'Generate a random %1$s', 'automatorwp-pro' ), '{random_email}' ),
            'log_label'         => sprintf( __( 'Generate a random %1$s', 'automatorwp-pro' ), '{random_email}' ),
            'options'           => array(
                'random_email' => array(
                    'default' => __( 'email', 'automatorwp-pro' ),
                    'fields' => array(
                        'username' => array(
                            'name' => __( 'Username:', 'automatorwp-pro' ),
                            'desc' => __( 'Accepts single or comma-separated list. Leave empty to get one.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                            'classes' => 'automatorwp-col-6'
                        ),
                        'username_length' => array(
                            'name' => __( 'Username length:', 'automatorwp-pro' ),
                            'desc' => __( 'Recommended from 1 to 64.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type'  => 'number',
                                'min'   => 1,
                                'max'   => 64
                            ),
                            'default' => 12,
                            'classes' => 'automatorwp-col-3'
                        ),
                        'username_characters' => array(
                            'name' => __( 'Type of characthers for username', 'automatorwp-pro' ),
                            'type' => 'multicheck',
                            'select_all_button' => false,
                            'options' => array(
                                'lowercase' => 'a-z (Lowercase)',
                                'uppercase' => 'A-Z (Uppercase)',
                                'number'    => '0-9 (Numbers)',
                            ),
                            'classes' => 'automatorwp-col-3',
                            'default' => 'lowercase',
                        ),
                        'domain' => array(
                            'name' => __( 'Domain:', 'automatorwp-pro' ),
                            'desc' => __( 'Accepts single or comma-separated list. Leave empty to get one.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                            'classes' => 'automatorwp-col-6'
                        ),
                        'domain_length' => array(
                            'name' => __( 'Domain length:', 'automatorwp-pro' ),
                            'desc' => __( 'Recommended from 1 to 64.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type'  => 'number',
                                'min'   => 1,
                                'max'   => 64
                            ),
                            'default' => 12,
                            'classes' => 'automatorwp-col-3'
                        ),
                        'domain_characters' => array(
                            'name' => __( 'Type of characthers for domain', 'automatorwp-pro' ),
                            'type' => 'multicheck',
                            'select_all_button' => false,
                            'options' => array(
                                'lowercase' => 'a-z (Lowercase)',
                                'uppercase' => 'A-Z (Uppercase)',
                                'number'    => '0-9 (Numbers)',
                            ),
                            'classes' => 'automatorwp-col-3',
                            'default' => 'lowercase',
                        ),
                        'tld' => array(
                            'name' => __( 'Extension:', 'automatorwp-pro' ),
                            'desc' => __( 'Accepts single or comma-separated list. Leave empty to get one.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '',
                        ),
                     ) )
            ),
            'tags'  => function_exists( 'automatorwp_generator_get_actions_random_email_tags' ) ? automatorwp_generator_get_actions_random_email_tags() : array(),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';
        
        // Shorthand
        $username = $action_options['username'];
        $domain = $action_options['domain'];
        $username_allowed_characters = $action_options['username_characters'];
        $domain_allowed_characters = $action_options['domain_characters'];
        $username_length = $action_options['username_length'];
        $domain_length = $action_options['domain_length'];
        $tld = $action_options['tld'];

        if ( empty( $username ) ){            
            $username = automatorwp_generator_random_string( $username_length, $username_allowed_characters );
        } else {
            // Get randomly username from list
            $username_list = explode( ",", $username );

            if (count( $username_list ) > 1 ) {
                $rand_key_username = array_rand( $username_list, 1 );
                $username = $username_list[$rand_key_username];
            }
        }

        if ( empty( $domain ) ){
            $domain = automatorwp_generator_random_string( $domain_length, $domain_allowed_characters );
        } else {
            // Get randomly domain from list
            $domain_list = explode( ",", $domain );

            if (count( $domain_list ) > 1 ) {
                $rand_key_domain = array_rand( $domain_list, 1 );
                $domain = $domain_list[$rand_key_domain];
            }
        }

        if ( empty( $tld ) ){
            $tld = automatorwp_generator_random_string( '3', array('lowercase') );
        } else {
            // Get randomly extension from list
            $extension_list = explode( ",", $tld );

            if (count( $extension_list ) > 1 ) {
                $rand_key_extension = array_rand( $extension_list, 1 );
                $tld = $extension_list[$rand_key_extension];
            }
        }
        
        $this->random_email = str_replace( ' ', '', $username . '@' . $domain . '.' . $tld );
        
        $this->result = sprintf( __( 'Random email created: %s', 'automatorwp-pro' ), $this->random_email );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );
		
		// Admin notice
		add_action( 'admin_notices', array( $this, 'admin_notice' ) );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Warn user if file was deleted
        if( ! function_exists( 'automatorwp_generator_get_actions_random_email_tags' ) ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo 
                    __( 'The generate a random email function was not found and may have been accidentally removed by a security plugin on your site.<br>', 'automatorwp-pro' ); ?>
                <?php echo
                    __( 'Please report this to the security plugin to add an exception to AutomatorWP and reinstall AutomatorWP to restore the deleted code.', 'automatorwp-pro' ); ?>
            </div>
        <?php endif;

    }
	
	 /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     */
    public function admin_notice( ) {
		
		// Warn user if file was deleted
        if( ! function_exists( 'automatorwp_generator_get_actions_random_email_tags' ) ) : ?>
            <div id="message-error" class="notice notice-error is-dismissible">
            <?php echo 
                    __( '<strong>AutomatorWP found an issue:</strong><br>', 'automatorwp' ); ?>
                <?php echo 
                    __( 'The generate a random email function was not found and may have been accidentally removed by a security plugin on your site.<br>', 'automatorwp-pro' ); ?>
                <?php echo
                    __( 'Please report this to the security plugin to add an exception to AutomatorWP and reinstall AutomatorWP to restore the deleted code.', 'automatorwp-pro' ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['random_email'] = ( isset( $this->random_email ) ? $this->random_email : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Generator_Generate_Email();