<?php
/**
 * Unroll User Course
 *
 * @package     AutomatorWP\Integrations\FluentCommunity\Actions\Unroll_User_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCommunity_Unroll_User_Course extends AutomatorWP_Integration_Action {

    public $integration = 'fluentcommunity';
    public $action = 'fluentcommunity_unroll_user_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Unroll user from a course', 'automatorwp-pro' ),
            'select_option'     => __( 'Unroll user from a <strong>course</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Course. */
            'edit_label'        => sprintf( __( 'Unroll user from %1$s', 'automatorwp-pro' ), '{course}' ),
            /* translators: %1$s: Course. */
            'log_label'         => sprintf( __( 'Unroll user from %1$s', 'automatorwp-pro' ), '{course}' ),
            'options'           => array(
                'course' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'course',
                    'option_default'    => __( 'course', 'automatorwp-pro' ),
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'action_cb'         => 'automatorwp_fluentcommunity_get_courses',
                    'options_cb'        => 'automatorwp_fluentcommunity_options_cb_course',
                    'placeholder'       => 'Select a course',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $course_id = $action_options['course'];
        $this->result = '';

        // Bail if empty course
        if ( empty( $course_id ) )
            return;

        try {
            \FluentCommunity\Modules\Course\Services\CourseHelper::leaveCourse( $course_id, $user_id );
        }
        catch (\Exception $e) {
            $this->result = __( 'The user could not be unrolled from the course.', 'automatorwp-pro' );
            return;
        }

        $this->result = __( 'User unrolled from the course successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }    

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_FluentCommunity_Unroll_User_Course();