<?php
/**
 * Set Role Space
 *
 * @package     AutomatorWP\Integrations\FluentCommunity\Actions\Set_Role_Space
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_FluentCommunity_Set_Role_Space extends AutomatorWP_Integration_Action {

    public $integration = 'fluentcommunity';
    public $action = 'fluentcommunity_set_role_space';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Set role to user in space', 'automatorwp-pro' ),
            'select_option'     => __( 'Set role to user in <strong>space</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Role. %1$s: Space. */
            'edit_label'        => sprintf( __( 'Set %1$s to user in %2$s', 'automatorwp-pro' ), '{role}', '{space}' ),
            /* translators: %1$s: Role. %1$s: Space. */
            'log_label'         => sprintf( __( 'Set %1$s to user in %2$s', 'automatorwp-pro' ), '{role}','{space}' ),
            'options'           => array(
                'space' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'space',
                    'option_default'    => __( 'space', 'automatorwp-pro' ),
                    'name'              => __( 'Space:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'action_cb'         => 'automatorwp_fluentcommunity_get_spaces',
                    'options_cb'        => 'automatorwp_fluentcommunity_options_cb_space',
                    'placeholder'       => 'Select a space',
                    'default'           => ''
                ) ),
                'role' => array(
                    'default' => __( 'choose role', 'automatorwp-pro' ),
                    'from' => 'role',
                    'fields' => array(
                        'role' => array(
                            'name' => __( 'Roles:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'member' => 'Member',
                                'moderator' => 'Moderator',
                                'admin'    => 'Admin',
                            ),
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $space_id = $action_options['space'];
        $role = $action_options['role'];
        $this->result = '';

        // Bail if empty space
        if ( empty( $space_id ) )
            return;

        $spaces_is_member = \FluentCommunity\App\Services\Helper::getUserSpaceIds( $user_id );

        if ( in_array( $space_id, $spaces_is_member ) ) {
            try {
                $user_data = \FluentCommunity\App\Models\SpaceUserPivot::bySpace( $space_id )
                    ->byUser( $user_id )
                    ->first();

                if ( $user_data ) {
                    if ($user_data->status == 'active') {
                        if ($role != $user_data->role) {
                            $user_data->role = $role;
                            $user_data->save();
                        } else {
                            $this->result = __( 'User already has the role selected.', 'automatorwp-pro' );
                            return;        
                        }
                    } else {
                        $this->result = __( 'User is not active.', 'automatorwp-pro' );
                        return;        
                    }
                } 
            }
            catch (\Exception $e) {
                $this->result = __( 'The role could not be set to user.', 'automatorwp-pro' );
                return;
            }
    
            $this->result = __( 'User role set successfully.', 'automatorwp-pro' );
        } else {
            $this->result = __( 'User does not belong to space.', 'automatorwp-pro' );
        }
/*
        
*/
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }    

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_FluentCommunity_Set_Role_Space();